package com.devplatform.admin.common.utils;

import java.io.IOException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import org.snmp4j.CommunityTarget;
import org.snmp4j.PDU;
import org.snmp4j.Snmp;
import org.snmp4j.event.ResponseEvent;
import org.snmp4j.mp.SnmpConstants;
import org.snmp4j.smi.Address;
import org.snmp4j.smi.GenericAddress;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.VariableBinding;
import org.snmp4j.transport.DefaultUdpTransportMapping;
/**
 * SNMP又称简单网络管理协议，由一组网络管理的标准组成，
 * 该协议能够支持网络管理系统，用以监测连接到网络上的设备是否有任何引起管理上关注的情况。
 * 简而言之，SNMP就是为不同种类、不同厂家生产、不同型号的设备，
 * 定义为一个统一的接口和协议，使得管理员可以是使用统一的方法来管理这些设备。
 * @author jzj
 * @date 2020/7/30 15:23
 */
public class SnmpGet {
  public static final int DEFAULT_VERSION = SnmpConstants.version2c;
  public static final String DEFAULT_PROTOCOL = "udp";
  public static final int DEFAULT_PORT = 161;
  public static final long DEFAULT_TIMEOUT = 3 * 1000;
  public static final int DEFAULT_RETRY = 3;

  /**
   * @param ip
   * @param community
   * @return
   */
  public static CommunityTarget createDefault(String ip, String community) {
    //生成目标地址对象
    Address address = GenericAddress.parse(DEFAULT_PROTOCOL + ":" + ip + "/" + DEFAULT_PORT);
    //CommunityTarget类实现了Target接口，用于SNMPv1和SNMPv2c这两个版本
    CommunityTarget target = new CommunityTarget();
    //设置snmp共同体
    target.setCommunity(new OctetString(community));
    //设置ip地址
    target.setAddress(address);
    //设置使用的snmp版本
    target.setVersion(DEFAULT_VERSION);
    // 超时时间
    target.setTimeout(DEFAULT_TIMEOUT);
    //重传次数
    target.setRetries(DEFAULT_RETRY);
    return target;
  }

  /**
   * 通过任意oid(对象标示符) ,get方式(get则是取具体的oid的值)请求服务器判断ip是否能ping通
   *
   * @return
   */
  public static boolean snmpPing(Long timeOut, String ip, String community, String oid) {

    CommunityTarget target = createDefault(ip, community);
    target.setTimeout(timeOut);
    Snmp snmp = null;
    try {
      //绑定要查询的OID
      PDU pdu = new PDU();
      // pdu.add(new VariableBinding(new OID(new int[]
      // {1,3,6,1,2,1,1,2})));
      pdu.add(new VariableBinding(new OID(oid)));

      //设定采取的协议
      DefaultUdpTransportMapping transport = new DefaultUdpTransportMapping();
      //创建SNMP对象，用于发送请求PDU
      snmp = new Snmp(transport);
      ////调用snmp中的listen()方法，启动监听进程，接收消息，由于该监听进程是守护进程，最后应调用close()方法来释放该进程
      snmp.listen();
      pdu.setType(PDU.GET);
      //调用 send(PDU pdu,Target target)发送pdu，返回一个ResponseEvent对象
      ResponseEvent respEvent = snmp.send(pdu, target);
      //通过ResponseEvent对象来获得SNMP请求的应答pdu，方法：public PDU getResponse()
      PDU response = respEvent.getResponse();

      if (response == null) {
      } else {
        //通过应答pdu获得mib信息（之前绑定的OID的值），方法：VaribleBinding get(int index)
        for (int i = 0; i < response.size(); i++) {
          VariableBinding vb = response.get(i);
          return true;
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
      return false;
    } finally {
      if (snmp != null) {
        try {
          snmp.close();
        } catch (IOException ex1) {
          snmp = null;
        }
      }
    }
    return false;
  }

  /**
   * 单个oid
   * @param ip
   * @param community
   * @param oid
   * @return
   */
  public static boolean snmpGet(String ip, String community, String oid) {

    CommunityTarget target = createDefault(ip, community);
    Snmp snmp = null;
    try {
      PDU pdu = new PDU();
      // pdu.add(new VariableBinding(new OID(new int[]
      // {1,3,6,1,2,1,1,2})));
      pdu.add(new VariableBinding(new OID(oid)));

      DefaultUdpTransportMapping transport = new DefaultUdpTransportMapping();
      snmp = new Snmp(transport);
      snmp.listen();
      pdu.setType(PDU.GET);
      ResponseEvent respEvent = snmp.send(pdu, target);
      PDU response = respEvent.getResponse();

      if (response == null) {
      } else {
        for (int i = 0; i < response.size(); i++) {
          VariableBinding vb = response.get(i);
          return true;
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
      return false;
    } finally {
      if (snmp != null) {
        try {
          snmp.close();
        } catch (IOException ex1) {
          snmp = null;
        }
      }
    }
    return false;
  }

  /**
   * .同步GET 多个OID（v2c）
   *
   * @param ip
   * @param community
   * @param oidList
   */
  public static List<String> snmpGetList(String ip, String community, List<String> oidList) {
    List<String> snmpList = new ArrayList<>();
    CommunityTarget target = SnmpGet.createDefault(ip, community);

    Snmp snmp = null;
    try {
      PDU pdu = new PDU();

      for (String oid : oidList) {
        pdu.add(new VariableBinding(new OID(oid)));
      }

      DefaultUdpTransportMapping transport = new DefaultUdpTransportMapping();
      transport.listen();
      snmp = new Snmp(transport);

      pdu.setType(PDU.GET);
      ResponseEvent respEvent = snmp.send(pdu, target);
      PDU response = respEvent.getResponse();

      if (response == null) {
      } else {
        for (int i = 0; i < response.size(); i++) {
          VariableBinding vb = response.get(i);
          snmpList.add(vb.getVariable() + "");
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
    } finally {
      if (snmp != null) {
        try {
          snmp.close();
        } catch (IOException ex1) {
          snmp = null;
        }
      }
    }
    return snmpList;
  }

  /**
   * @param ip
   * @return
   */
  public static List<String> getSnmpLinux(String ip, String community) {
    List<String> oidList = new ArrayList<String>();
    // 系统基本信息
    oidList.add(".1.3.6.1.2.1.1.1.0");
    // 系统名称
    oidList.add(".1.3.6.1.2.1.1.5.0");
    // 内存大小
    oidList.add(".1.3.6.1.2.1.25.2.2.0");
    // 磁盘占用率
    oidList.add(".1.3.6.1.4.1.2021.9.1.9.1");
    // 磁盘总数
    oidList.add(".1.3.6.1.4.1.2021.9.1.6.1");

    return SnmpGet.snmpGetList(ip, community, oidList);
  }

  public static void main(String[] args) throws UnknownHostException {
    String ip = "10.20.2.139";
    String community = "public";
    System.out.println(SnmpGet.getSnmpLinux(ip, community));
  }
}
