package com.devplatform.admin.common.utils;

import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.streaming.*;
import org.apache.poi.xssf.usermodel.XSSFClientAnchor;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * 生成导出Excel文件对象
 *
 * @author Muceball-laptop
 * @version 1.0
 * @date 2021/1/27 16:25
 */
public class XlsExportUtil {
    private static final Pattern linePattern = Pattern.compile("_(\\w)");
    private static final Pattern humpPattern = Pattern.compile("[A-Z]");
    private static final Logger logger = LoggerFactory.getLogger(XlsExportUtil.class);
    private final String xlsFileName;
    private final SXSSFWorkbook workbook;
    private final SXSSFSheet sheet;
    private final SXSSFDrawing drawing;
    private final Map<String, CellStyle> styles;
    private SXSSFRow row;
    private int rowNum = 0;
    private String path;
    private List<Map<String, String>> titles;
    private Set<String> blacklist;
    private Map<String, String> titleTranslate;
    private HashMap<String, HashMap<Object, String>> dataTranslate;

    /**
     * 初始化Excel
     *
     * @param fileName 导出文件名
     * @param path     导出文件路径
     */
    public XlsExportUtil(String fileName, String path) {
        //文件名
        this.xlsFileName = fileName;
        //路径
        this.path = path;
        //默认超过100行数据就缓存到硬盘
        this.workbook = new SXSSFWorkbook(100);
        this.styles = createStyles(workbook);
        this.sheet = workbook.createSheet();
        this.drawing = sheet.createDrawingPatriarch();
    }

    /**
     * 导出Excel文件
     */
    @SuppressWarnings("all")
    public void exportXlS() throws Exception {
        if (StringUtils.isNotBlank(path) && !path.endsWith("/")) {
            path = path + "/";
        }
        File file = new File(path);
        if (!file.exists() && !file.isDirectory()) {
            logger.info("创建文件夹：" + path);
            file.mkdirs();
        }
        try (FileOutputStream out = new FileOutputStream(path + xlsFileName + ".xlsx")) {
            workbook.write(out);
        } finally {
            if (workbook != null) {
                try {
                    workbook.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            workbook.dispose();
        }
    }

    /**
     * 导出Excel文件
     *
     * @param titles 标题数组
     * @param data   数据列表
     */
    public void exportXlS(List<Map<String, String>> titles, List<HashMap<String, Object>> data) throws Exception {
        setMapTitle(titles);
        setData(data);
        exportXlS();
    }

    /**
     * 设置标题
     *
     * @param titles 标题数组
     */
    public void setTitle(List<String> titles) throws Exception {
        createRow();
        int cell = 0;
        for (String name : titles) {
            name = lineToHump(name);
            setStyle(row, cell);
            setCell(cell++, name, styles.get("header"));
        }
    }

    /**
     * 设置标题
     *
     * @param titles 标题数组
     */
    public void setTitle(String... titles) throws Exception {
        createRow();
        int cell = 0;
        for (String name : titles) {
            name = lineToHump(name);
            setStyle(row, cell);
            setCell(cell++, name, styles.get("header"));
        }
    }

    /**
     * 设置标题
     * 从数据库查询表头信息设置导出字段
     * blacklist：导出黑名单(过滤掉不需要导出的字段)需要通过setBlacklist()方法设置
     * titleTranslate：字段翻译器(数据通过设置好的对应关系导出)需要通过setDataTranslate()方法设置
     *
     * @param titles 标题数组
     */
    public void setMapTitle(List<Map<String, String>> titles) throws Exception {
        this.titles = titles;
        createRow();
        int cell = 0;
        Iterator<Map<String, String>> iterator = titles.iterator();
        while (iterator.hasNext()) {
            Map<String, String> title = iterator.next();
            String name = title.get("columnName");
            String comment = title.get("columnComment");
            //过滤不导出的字段
            if (blacklist != null && blacklist.contains(name)) {
                iterator.remove();
                continue;
            }
            //手动设置字段对应管理
            if (titleTranslate != null && titleTranslate.containsKey(name)) {
                comment = titleTranslate.get(name);
            }
            setStyle(row, cell);
            //有备注导出备注，无备注导出原字段名
            if (StringUtils.isNotBlank(comment)) {
                setCell(cell++, comment, styles.get("header"));
            } else {
                setCell(cell++, name, styles.get("header"));
            }
            name = lineToHump(name);
            title.put("columnName", name);
        }
    }

    /**
     * 设置数据
     * <p>
     * 注意：使用数据库查询表头的方式适用
     * 使用setTitle方法设置表头的情况，请使用createRow()和setCell()手动实现设置值
     * <p>
     * 使用{@link XlsExportUtil#setTitle(String...)} 或者是{@link XlsExportUtil#setTitle(List)}方法设置表头的情况，
     * <br>请使用{@link XlsExportUtil#createRow(int)}和{@link XlsExportUtil#setCell(int, String)}手动实现设置值
     *
     * @param data 数据列表
     */
    public void setData(List<HashMap<String, Object>> data) throws Exception {
        //设置数据
        for (HashMap<String, Object> obj : data) {
            createRow();
            int cell = 0;
            //map模式的title导出
            for (Map<String, String> title : titles) {
                String type = title.get("dataType");
                String column = title.get("columnName");
                Object value = obj.get(column);
                try {
                    if (dataTranslate != null && dataTranslate.containsKey(column)) {
                        setCell(cell++, dataTranslate.get(column).get(value));
                        continue;
                    }
                    setCell(cell++, type, value);
                } catch (Exception ex) {
                    setCell(cell++, "default value");
                    logger.info("Set index value [" + row.getRowNum() + "," + cell + "] error: " + ex.toString());
                }
            }
        }
    }

    /**
     * 增加一行
     *
     * @param index 行号
     */
    public void createRow(int index) {
        this.row = this.sheet.createRow(index);
    }

    /**
     * 增加一行
     */
    public void createRow() {
        this.row = this.sheet.createRow(rowNum++);
    }

    /**
     * 设置单元格
     *
     * @param index      列号
     * @param value      单元格填充值
     * @param commentStr 批注信息
     */
    public void setCell(int index, String value, String commentStr) {
        SXSSFCell cell = this.row.createCell(index);
        cell.setCellType(CellType.STRING);
        cell.setCellValue(value);
        Comment comment = cell.getCellComment();
        if (comment == null) {
            //批注为3行4列大小(rowNum减去1是因为创建一行后rowNum会自动加一)
            comment = drawing.createCellComment(new XSSFClientAnchor(0, 0, 0, 0, index, rowNum - 1, index + 3, rowNum + 2));
        }
        // 输入批注信息
        comment.setString(new XSSFRichTextString(commentStr));
        // 添加作者
        comment.setAuthor("Suntray");
        // 将批注添加到单元格对象中
        cell.setCellComment(comment);
    }

    /**
     * 设置单元格
     *
     * @param index 列号
     * @param value 单元格填充值
     */
    public void setCell(int index, String value, CellStyle style) {
        SXSSFCell cell = this.row.createCell(index);
        cell.setCellType(CellType.STRING);
        cell.setCellValue(value);
        cell.setCellStyle(style);
    }

    /**
     * 设置单元格
     *
     * @param index 列号
     * @param value 单元格填充值
     */
    public void setCell(int index, String value) {
        SXSSFCell cell = this.row.createCell(index);
        cell.setCellStyle(styles.get("data"));
        cell.setCellType(CellType.STRING);
        cell.setCellValue(value);
    }

    /**
     * 设置单元格
     *
     * @param index 列号
     * @param value 单元格填充值
     */
    private void setCell(int index, String type, Object value) {
        SXSSFCell cell = this.row.createCell(index);
        cell.setCellStyle(styles.get("data"));
        cell.setCellType(CellType.STRING);
        if (value == null) {
            cell.setCellValue("");
        } else if (value instanceof String) {
            cell.setCellValue((String) value);
        } else if (value instanceof Integer) {
            cell.setCellValue((Integer) value);
        } else if (value instanceof Long) {
            cell.setCellValue((Long) value);
        } else if (value instanceof Double) {
            cell.setCellValue((Double) value);
        } else if (value instanceof Float) {
            cell.setCellValue((Float) value);
        } else if (value instanceof Date) {
//            if (BaseConstant.DATE.equals(type)) {
//                cell.setCellValue(DateUtils.format((Date) value, "yyyy-MM-dd"));
//            } else if (BaseConstant.DATETIME.equals(type)) {
//                cell.setCellValue(DateUtils.format((Date) value, "yyyy-MM-dd HH:mm:ss"));
//            } else {
//                cell.setCellValue((Date) value);
//            }
        } else if (value instanceof BigDecimal) {
            double doubleVal = ((BigDecimal) value).doubleValue();
            cell.setCellValue(doubleVal);
        } else {
            cell.setCellValue(value.toString());
        }
    }

    /**
     * 设置单元格(自定义类型)
     *
     * @param index    列号
     * @param value    单元格填充值
     * @param cellType 单元格格式
     */
    private void setCell(int index, String value, CellType cellType) {
        SXSSFCell cell = this.row.createCell(index);
        cell.setCellType(cellType);
        cell.setCellValue(value);
    }

    /**
     * 下划线转驼峰
     */
    public static String lineToHump(String str) {
        str = str.toLowerCase();
        Matcher matcher = linePattern.matcher(str);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(sb, matcher.group(1).toUpperCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * 驼峰转下划线(简单写法，效率低于{@link #humpToLine(String)})
     */
    public static String humpToLine1(String str) {
        return str.replaceAll("[A-Z]", "_$0").toLowerCase();
    }

    /**
     * 驼峰转下划线,效率比上面高
     */
    public static String humpToLine(String str) {
        Matcher matcher = humpPattern.matcher(str);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(sb, "_" + matcher.group(0).toLowerCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * 设置不导出的字段
     *
     * @param blacklist 不需要导出字段列表
     */
    public void setBlacklist(Set<String> blacklist) {
        this.blacklist = blacklist;
    }

    /**
     * 设置不导出的字段
     *
     * @param blacklist 不需要导出字段列表
     */
    public void setBlacklist(String... blacklist) {
        if (blacklist == null || blacklist.length == 0) {
            return;
        }
        this.blacklist = Stream.of(blacklist).collect(Collectors.toSet());
    }

    /**
     * 获取下载文件带路径名称
     *
     * @return 文件名
     */
    public String getFullFileName() {
        return path + xlsFileName + ".xls";
    }

    /**
     * 设置标题翻译器
     * 用于指定字段对应的导出备注（excel标题）
     *
     * @param titleTranslate 翻译内容
     */
    public void setTitleTranslate(Map<String, String> titleTranslate) {
        this.titleTranslate = titleTranslate;
    }

    /**
     * 设置数据翻译器
     * 翻译数据中的字段：
     * 例如把 sex字段中的 1翻译为男  2翻译为女
     *
     * @param dataTranslate 数据翻译对应关系
     */
    public void setDataTranslate(HashMap<String, HashMap<Object, String>> dataTranslate) {
        HashMap<String, HashMap<Object, String>> map = new HashMap<>(16);
        for (Map.Entry<String, HashMap<Object, String>> entry : dataTranslate.entrySet()) {
            map.put(lineToHump(entry.getKey()), entry.getValue());
        }
        this.dataTranslate = map;
    }

    /**
     * 创建表格样式
     *
     * @param wb 工作薄对象
     * @return 样式列表
     */
    private Map<String, CellStyle> createStyles(Workbook wb) {
        // 写入各条记录,每条记录对应excel表中的一行
        Map<String, CellStyle> styles = new HashMap<String, CellStyle>();
        CellStyle style = wb.createCellStyle();
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        style.setBorderRight(BorderStyle.THIN);
        style.setRightBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setBorderLeft(BorderStyle.THIN);
        style.setLeftBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setBorderTop(BorderStyle.THIN);
        style.setTopBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setBorderBottom(BorderStyle.THIN);
        style.setBottomBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
        Font dataFont = wb.createFont();
        dataFont.setFontName("Arial");
        dataFont.setFontHeightInPoints((short) 10);
        style.setFont(dataFont);
        styles.put("data", style);

        style = wb.createCellStyle();
        style.cloneStyleFrom(styles.get("data"));
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        style.setFillForegroundColor(IndexedColors.GREY_50_PERCENT.getIndex());
        style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        Font headerFont = wb.createFont();
        headerFont.setFontName("Arial");
        headerFont.setFontHeightInPoints((short) 10);
        headerFont.setBold(true);
        headerFont.setColor(IndexedColors.WHITE.getIndex());
        style.setFont(headerFont);
        styles.put("header", style);

        style = wb.createCellStyle();
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        Font totalFont = wb.createFont();
        totalFont.setFontName("Arial");
        totalFont.setFontHeightInPoints((short) 10);
        style.setFont(totalFont);
        styles.put("total", style);

        return styles;
    }


    /**
     * 创建表格样式
     */
    public void setStyle(Row row, int column) {
        // 设置列宽
        sheet.setColumnWidth(column, (int) ((16.72) * 256));
        row.setHeight((short) (14 * 20));
    }


    /**
     * 导出Excel文件
     *
     * @param titles     标题数组
     * @param dataMap    数据列表
     * @param commentMap 批注信息
     */
    public void exportXlSList(String[] titles, Map<Integer, Map<Integer, Object>> dataMap, Map<Integer, Map<Integer, Object>> commentMap) throws Exception {
        //写入标题
        setTitle(titles);
        //写入数据和批注
        setDataList(dataMap, commentMap);
        //生成文件
        exportXlS();
    }

    /**
     * 设置excel数据
     *
     * @param dataMap    数据信息
     * @param commentMap 批注信息
     */
    public void setDataList(Map<Integer, Map<Integer, Object>> dataMap, Map<Integer, Map<Integer, Object>> commentMap) {
        //设置数据
        for (int i = 1; i <= dataMap.size(); i++) {
            //每一行的数据
            Map<Integer, Object> map = dataMap.get(i);
            //创建一行
            createRow();
            //遍历每一行的每一列
            for (int j = 0; j < map.size(); j++) {
                //每个单元的数据
                Object value = map.get(j);
                //判断当前行是否有批注
                if (commentMap!=null && commentMap.containsKey(i) && commentMap.get(i).containsKey(j)) {
                    //存在说明有批注，将当前单元格设置批注信息
                    setCell(j, (String) value, (String) commentMap.get(i).get(j));
                } else {
                    //没有批注信息则仅设置普通值
                    setCell(j, (String) value);
                }
            }
        }
    }

    public void exportXlSX(String[] titles, List<Map<Integer, Object>> dataList) throws Exception {
        //写入标题
        setTitle(titles);
        //写入数据和批注
        setDataList(dataList);
        //生成文件
        exportXlS();
    }

    public void setDataList(List<Map<Integer, Object>> dataList) {
        //设置数据
        for (int i = 0; i < dataList.size(); i++) {
            //每一行的数据
            Map<Integer, Object> map = dataList.get(i);
            //创建一行
            createRow();
            //遍历每一行的每一列
            for (int j = 0; j < map.size(); j++) {
                //每个单元的数据
                Object value = map.get(j);
                setCell(j, (String) value);
            }
        }
    }
    /**
     * 根据标题+数据+总列数导出excel
     * @param titles    标题
	 * @param dataList  数据，map的key为列号
	 * @param rowSize   总列数
     * @return void
     * @author dhl
     * @date 2021/3/5 0005 11:42
     */
    public void exportXlSXByValIndex(String[] titles, List<HashMap<String, Object>> dataList,Integer rowSize) throws Exception {
        //写入标题
        setTitle(titles);
        //写入数据和批注
        setDataListByValIndex(dataList,rowSize);
        //生成文件
        exportXlS();
    }

    /**
     * 根据数据创建单元格
     * @param dataList  数据，map的key为列号
	 * @param rowSize   总列数
     * @return void
     * @author dhl
     * @date 2021/3/5 0005 11:43
     */
    public void setDataListByValIndex(List<HashMap<String, Object>> dataList,Integer rowSize) {
        //设置数据
        for (int i = 0; i < dataList.size(); i++) {
            //每一行的数据
            Map<String, Object> map = dataList.get(i);
            //创建一行
            createRow();
            //遍历每一行的每一列
            for (int j = 0; j < rowSize; j++) {
                //每个单元的数据
                Object value = map.get(j+"");
                setCell(j, "",value);
            }
        }
    }

}
