package com.devplatform.admin.modules.liresource.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.devplatform.admin.common.annotation.SysLogMethod;
import com.devplatform.admin.common.utils.AbstractController;
import com.devplatform.admin.common.utils.Constants;
import com.devplatform.admin.modules.liresource.bean.LiResource;
import com.devplatform.admin.modules.liresource.bean.LiStation;
import com.devplatform.admin.modules.liresource.bean.LiStationMap;
import com.devplatform.admin.modules.liresource.model.LiResourceModel;
import com.devplatform.admin.modules.liresource.model.LiStationMapModel;
import com.devplatform.admin.modules.liresource.service.LiResourceService;
import com.devplatform.admin.modules.liresource.service.LiStationMapService;
import com.devplatform.admin.modules.liresource.service.LiStationService;
import com.devplatform.admin.modules.sysdictionary.bean.SysDictionary;
import com.devplatform.admin.modules.sysdictionary.service.SysDictionaryService;
import com.devplatform.common.base.annotation.SysLog;
import com.devplatform.common.base.validator.ValidatorUtils;
import com.devplatform.common.util.R;
import com.devplatform.common.util.StringUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Administrator
 */
@Api(tags = {"资源点管理表接口"})
@RestController
@RequestMapping("/liResource")
public class LiResourceController extends AbstractController {
    @Autowired
    private LiResourceService liResourceService;
    @Autowired
    private LiStationService liStationService;
    @Autowired
    private LiStationMapService liStationMapService;
    @Autowired
    private SysDictionaryService sysDictionaryService;

    /**
     * 列表页面列表数据获取
     *
     * @param model 承接对象
     * @return
     */
    @ApiOperation(value = "根据条件获取资源点管理表分页数据列表", notes = "根据条件获取资源点管理表分页数据列表")
    @ApiImplicitParam(name = "model", value = "参数", required = true, dataType = "LiResourceModel")
    @PostMapping("/list")
    public R list(@RequestBody LiResourceModel model) {
        if (StringUtils.isBlank(model.getSort())) {
            model.setSort("create_time");
            model.setOrder("DESC");
        }
        List<LiResource> list = liResourceService.queryPageByList(model);
        return R.ok().put("page", getMyPage(list, model));
    }

    /**
     * 添加
     *
     * @param bean 资源点管理表对象
     * @return
     */
    @ApiOperation(value = "新增资源点管理表数据", notes = "新增资源点管理表数据")
    @PostMapping("/save")
    public R save(@RequestBody @ApiParam(value = "资源点管理表实体对象", required = true) LiResource bean) {
        ValidatorUtils.validateEntity(bean);

        if (StringUtils.isNotBlank(bean.getType()) && StringUtils.isNotBlank(bean.getCodeType())) {
            bean.setCode(generateCode(bean.getCodeType(), bean.getType()));
        } else {
            return R.error().put("msg", "缺失必要参数！");
        }
        bean.setCreateUserId(getUserId());
        bean.setCreateTime(new Date());
        bean.setOperatorId(getUserId());
        bean.setOperatorName(getUser().getUsername());
        liResourceService.save(bean);
        return R.ok().put("bean", bean);
    }

    /**
     * 生成资源点编码
     *
     * @param codeType 编码前缀
     * @param type     资源点类型
     * @return
     */
    public String generateCode(String codeType, String type) {
        Map<String, Object> params = new HashMap(5);
        params.put("type", type);
        params.put("deleted", 0);
        Integer code = liResourceService.count(new QueryWrapper<LiResource>().allEq(params));
        return codeType + "-" + String.format("%04d", code + 1);
    }

    /**
     * 修改
     *
     * @param bean 资源点管理表对象
     * @return
     */
    @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "修改资源点信息")
    @ApiOperation(value = "修改资源点管理表数据", notes = "修改资源点管理表数据")
    @PostMapping("/update")
    public R update(@RequestBody @ApiParam(value = "资源点管理表实体对象", required = true) LiResource bean) {
        ValidatorUtils.validateEntity(bean);
        bean.setUpdateUserId(getUserId());
        bean.setUpdateTime(new Date());
        // 判断父节点是否为空，为空则把数据库中的也清空
        liResourceService.update(
                bean,
                new UpdateWrapper<LiResource>()
                        .eq(StringUtil.checkNotNull(bean.getId()), "id", bean.getId()));
        return R.ok().put("bean", bean);
    }

    /**
     * 修改
     *
     * @param list 资源点管理表对象
     * @return
     */
    @SysLog("批量修改资源点管理表")
    @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "批量修改资源点信息")
    @ApiOperation(value = "批量修改资源点管理表数据", notes = "批量修改资源点管理表数据")
    @PostMapping("/updateList")
    public R updateList(@RequestBody @ApiParam(value = "资源点管理表实体对象", required = true) List<LiResource> list) {
        if (list != null && list.size() > 0) {
            List<LiResource> updateList = new ArrayList<LiResource>();
            Date now = new Date();
            list = list.parallelStream().distinct().collect(Collectors.toList());
//            Set<String> tips = new HashSet<>(32);
//            deduplication(list, list.size(), tips);
//            if (!tips.isEmpty()) {
//                StringBuilder builder = new StringBuilder();
//                for (String tip : tips) {
//                    builder.append(tip).append("<br>");
//                }
//                return R.ok().put("message", builder.toString());
//            }
            for (LiResource resource : list) {
                if ("save".equals(resource.getAddOrUpdate())) {
                    resource.setId(null);
                    resource.setCreateUserId(getUserId());
                    resource.setCreateTime(now);
//                    if (StringUtils.isNotBlank(resource.getType()) && StringUtils.isNotBlank(resource.getCodeType())) {
//                        resource.setCode(generateCode(resource.getCodeType(), resource.getType()));
//                    }
                    liResourceService.save(resource);
                } else {
                    resource.setUpdateUserId(getUserId());
                    resource.setUpdateTime(now);
                    updateList.add(resource);
                }
            }
            if (updateList != null && updateList.size() > 0) {
                liResourceService.updateBatchById(updateList);
            }
        }
        return R.ok();
    }

    /**
     * 添加去重信息
     *
     * @param list
     * @param size
     * @param tips
     */
    private void deduplication(List<LiResource> list, int size, Set<String> tips) {
        for (int i = 0; i < size; i++) {
            for (int j = i; j < size; j++) {
                LiResource base = list.get(i);
                LiResource resource = list.get(j);
                if (base.getId() == null || base.getId().equals(resource.getId())) {
                    //没有id的或者是自己就直接跳过
                    continue;
                }
                addTips(tips, base, resource);
            }
        }
        //查询所有的数据
        List<LiResource> list1 = liResourceService.list(new LambdaQueryWrapper<LiResource>().eq(LiResource::getDeleted, 0));
        for (LiResource resource : list1) {
            for (LiResource liResource : list) {
                if (resource.getId().equals(liResource.getId())) {
                    continue;
                }
                addTips(tips, resource, liResource);
            }
        }
    }

    /**
     * 添加提示，防止数据重复（数据不全的直接跳过校验）
     *
     * @param tips     提示集合
     * @param base     全部的数据
     * @param resource 需要比较的值
     */
    private void addTips(Set<String> tips, LiResource base, LiResource resource) {
        switch (base.getType()) {
            //枪机 球机 安检机
            case "b6af764f2a6e454490a6b1b3c9057e57":
            case "ece0b8b2db27411886254e81134988a3":
            case "303310efddb34a2e9bf269bdff8a7dc5":
                if (base.getAddressCode() == null || base.getPort() == null || base.getByx1() == null || base.getDeleted() == 1) {
                    break;
                }
                if (Objects.equals(base.getAddressCode(), resource.getAddressCode()) &&
                        Objects.equals(base.getPort(), resource.getPort()) && Objects.equals(base.getByx1(), resource.getByx1())) {
                    //添加到重复提示
                    StringBuilder builder = new StringBuilder()
                            .append("地址码:").append(base.getAddressCode())
                            .append(" 端口:").append(base.getPort())
                            .append(" 通道号:").append(base.getByx1());
                    tips.add(builder.toString());
                }
                break;
            default:
                if (base.getAddressCode() == null || base.getPort() == null || base.getDeleted() == 1) {
                    break;
                }
                if (Objects.equals(base.getAddressCode(), resource.getAddressCode()) &&
                        Objects.equals(base.getPort(), resource.getPort())) {
                    //添加到重复提示
                    StringBuilder builder = new StringBuilder()
                            .append("地址码:").append(base.getAddressCode())
                            .append(" 端口:").append(base.getPort());
                    tips.add(builder.toString());
                }
        }
    }

    private boolean checkDuplicate(List<LiResource> list, LiResource liResource) {
        for (LiResource resource : list) {
            if (!resource.getId().equals(liResource.getId()) && resource.equals(liResource)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 删除资源点前校验
     *
     * @param id 对象主键
     * @return
     */
    @ApiOperation(value = "删除资源点前校验", notes = "删除资源点前校验")
    @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
    @GetMapping("/validateDeleteResource")
    public R validateDeleteResource(String id, String stationId) {
        return R.ok();
    }

    /**
     * 根据ID获取资源点管理表对象
     *
     * @param id 对象主键
     * @return
     */
    @SysLogMethod(operation = "查看", blockName = "系统管理", detail = "查看资源点详情")
    @ApiOperation(value = "根据ID获取资源点管理表对象", notes = "根据ID获取资源点管理表对象")
    @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
    @GetMapping("/getId")
    public R getId(String id, String stationId) {
        LiResource bean = liResourceService.getBeanById(id, stationId);
        return R.ok().put("bean", bean);
    }

    /**
     * 根据ID获取资源点管理表对象
     *
     * @param ids 资源点管理表对象主键数组
     * @return
     */
    @SysLogMethod(operation = "删除", blockName = "系统管理", detail = "删除资源点数据")
    @ApiOperation(value = "根据ID批量删除资源点管理表数据", notes = "根据ID批量删除资源点管理表数据")
    @ApiImplicitParam(name = "ids", value = "主键数组", required = true, dataType = "String[]")
    @PostMapping("/delete")
    public R delete(@RequestBody String[] ids) {
        //用来暂存所有要删除的对象
        List<LiResource> beans = new ArrayList<LiResource>();
        //遍历ids数组
        for(String id: ids){
            LiResource temp = new LiResource();
            temp.setId(id);
            //将数据标记为删除
            temp.setDeleted(1);
            temp.setUpdateUserId(getUserId());
            temp.setUpdateTime(new Date());
            //将要删除的对象存放在list中
            beans.add(temp);
        }
        //调用批量更新操作
        liResourceService.updateBatchById(beans);
        return R.ok();
    }

    /**
     * 获取站点-资源点树
     * @param model 承接对象
     * @return
     */
    @ApiOperation(value = "获取站点-资源点树", notes = "获取站点-资源点树")
    @ApiImplicitParam(name = "model", value = "参数", required = true, dataType = "LiResourceModel")
    @PostMapping("/getTreeList")
    public R getTreeList(@RequestBody LiResourceModel model) {
        // 默认只查询一个站点
        List<Map<String, Object>> tree = new ArrayList<>();
        tree = getStationTreeList(model);
        return R.ok().put("tree", tree);
    }

    private List<Map<String, Object>> getStationTreeList(LiResourceModel model) {
        List<Map<String, Object>> tree = new ArrayList<>();
        LiStation station = liStationService.getById(model.getStationId());
        Map<String, Object> root = JSONObject.parseObject(JSONObject.toJSONString(station), Map.class);
        root.put("name", station.getStationName());
        root.put("key", UUID.randomUUID());
        root.put("level", 3);

        // 查询该站点下站层地图
        LiStationMapModel mapModel = new LiStationMapModel();
        mapModel.setStationId(model.getStationId());
        List<LiStationMap> list = liStationMapService.getTreeList(mapModel);

        // 查询资源点类型典项
        List<SysDictionary> dicList = new ArrayList<>();
        dicList = sysDictionaryService.queryByPid(Constants.RESOURCE_TYPE, model.getStationId());

        // 查询该站点下所有资源点
        List<LiResource> resourceList = liResourceService.getTreeList(model);

        List<Map<String, Object>> nodes = new ArrayList<>();
        for (LiStationMap liStationMap : list) {
            Map<String, Object> mapStationMap = JSONObject.parseObject(JSONObject.toJSONString(liStationMap), Map.class);
            mapStationMap.put("key", UUID.randomUUID());
            mapStationMap.put("level", 4);

            List<Map<String, Object>> childrenStationMap = new ArrayList<>();
            for (SysDictionary dic : dicList) {
                Map<String, Object> mapDic = JSONObject.parseObject(JSONObject.toJSONString(dic), Map.class);
                // 两个id拼接保证id的唯一性
                mapDic.put("id", liStationMap.getId() + dic.getId());
                mapDic.put("code", dic.getValue());
                mapDic.put("stationId", liStationMap.getStationId());
                mapDic.put("tierId", liStationMap.getId());
                mapDic.put("key", UUID.randomUUID());
                mapDic.put("level", 5);

                List<Map<String, Object>> childrenDic = new ArrayList<>();
                Iterator<LiResource> iterator = resourceList.iterator();
                while (iterator.hasNext()) {
                    LiResource resource = iterator.next();
                    if (dic.getId().equals(resource.getType()) && liStationMap.getId().equals(resource.getTierId())) {
                        Map<String, Object> mapResource = JSONObject.parseObject(JSONObject.toJSONString(resource), Map.class);
                        mapResource.put("level", 6);
                        mapResource.put("active", false);
                        // 拖拽的唯一标识
                        mapResource.put("key", resource.getId());
                        mapResource.put("installDate",resource.getInstallDate());
                        mapResource.put("maintenanceDate",resource.getMaintenanceDate());
                        childrenDic.add(mapResource);
                        iterator.remove();
                    }
                }
                mapDic.put("children", childrenDic);
                /** 如果该设备类型下无资源点，不展示*/
                if (childrenDic != null && childrenDic.size() > 0) {
                    childrenStationMap.add(mapDic);
                }
            }
            mapStationMap.put("children", childrenStationMap);
            nodes.add(mapStationMap);
        }
        root.put("children", nodes);
        tree.add(root);
        return tree;
    }

    /**
     * 根据站点id查询站点地图
     *
     * @param model 承接对象 stationId 当前站点id subCode 子系统编码
     * @return
     */
    @ApiOperation(value = "根据站点id查询站点地图", notes = "根据站点id查询站点地图")
    @ApiImplicitParam(name = "model", value = "参数", required = true, dataType = "LiStationMapModel")
    @PostMapping("/getStationMapCodeTreeList")
    public R getStationMapCodeTreeList(@RequestBody LiStationMapModel model) {
        List<Map<String, Object>> nodes = new ArrayList<>();
        // 查询当前站点相应所有层级
        List<LiStationMap> list = liStationMapService.getTreeList(model);
        if (list != null && list.size() > 0) {
            for (int i = 0; i < list.size(); i++) {
                Map<String, Object> map = new LinkedHashMap<String, Object>();
                map.put("id", list.get(i).getId());
                /** 站层名称*/
                map.put("name", list.get(i).getName());
                LiResourceModel resourceModel = new LiResourceModel();
                /** 当前站点id*/
                resourceModel.setStationId(model.getStationId());
                /** 站层id*/
                resourceModel.setTierId(list.get(i).getId());
                // 查询当前站层下，所有的资源点
                List<LiResource> resourceList = liResourceService.getTreeList(resourceModel);
                if (resourceList != null && resourceList.size() > 0) {
                    List<Map<String, Object>> childrenDic = new ArrayList<>();
                    for (int j = 0; j < resourceList.size(); j++) {
                        Map<String, Object> mapChildren = new LinkedHashMap<String, Object>();
                        mapChildren.put("id", resourceList.get(j).getId());
                        /** 资源点名称*/
                        mapChildren.put("name", resourceList.get(j).getName());
                        /** 资源点类型*/
                        mapChildren.put("type", resourceList.get(j).getType());
                        /** 站点名称*/
                        mapChildren.put("stationName", resourceList.get(j).getStationName());
                        /** 资源点部署位置*/
                        mapChildren.put("deployLocation", resourceList.get(j).getDeployLocation());
                        /** 地址码*/
                        mapChildren.put("addressCode", resourceList.get(j).getAddressCode());
                        /** 端口号*/
                        mapChildren.put("port", resourceList.get(j).getPort());
                        /** 用户名*/
                        mapChildren.put("username", resourceList.get(j).getUsername());
                        /** 密码*/
                        mapChildren.put("password", resourceList.get(j).getPassword());
                        childrenDic.add(mapChildren);
                    }
                    /** 把当前站层下所有的资源点，绑定到站层上*/
                    map.put("children", childrenDic);
                }
                /** 把当前站层所有数据，让如树中*/
                nodes.add(map);
            }
        }
        return R.ok().put("nodes", nodes);
    }
}
