package com.devplatform.admin.modules.sys.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.devplatform.admin.common.annotation.SysLogMethod;
import com.devplatform.admin.common.utils.AbstractController;
import com.devplatform.admin.common.utils.Aes;
import com.devplatform.admin.modules.sys.bean.SysRoleEntity;
import com.devplatform.admin.modules.sys.bean.SysUserEntity;
import com.devplatform.admin.modules.sys.form.PasswordForm;
import com.devplatform.admin.modules.sys.model.SysUserModel;
import com.devplatform.admin.modules.sys.service.SysRoleService;
import com.devplatform.admin.modules.sys.service.SysUserRoleService;
import com.devplatform.admin.modules.sys.service.SysUserService;
import com.devplatform.common.base.annotation.SysLog;
import com.devplatform.common.base.validator.Assert;
import com.devplatform.common.util.Constant;
import com.devplatform.common.util.R;
import io.swagger.annotations.*;
import lombok.SneakyThrows;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 系统用户
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2016年10月31日 上午10:40:10
 */
@Api(tags = {"系统用户接口"})
@RestController
@RequestMapping("/sys/user")
public class SysUserController extends AbstractController {
    @Autowired
    private SysRoleService sysRoleService;
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private SysUserRoleService sysUserRoleService;

    /**
     * 所有用户列表
     */
    @ApiOperation(value = "根据条件获取分页数据列表", notes = "根据条件获取分页数据列表")
    @ApiImplicitParam(
            name = "params",
            value = "查询参数",
            required = true,
            paramType = "query",
            dataType = "Map<String, Object>")
    @GetMapping("/list")
    public R list(@RequestParam Map<String, Object> params) {
        // 只有超级管理员，才能查看所有管理员列表
        SysUserModel model = new SysUserModel();
        String username = (String) params.get("username");
        model.setUsername(username);
        String orgId = (String) params.get("orgId");
        model.setOrgId(orgId);
        String stationId = (String) params.get("stationId");
        model.setStationId(stationId);
        String page = (String) params.get("page");
        String limit = (String) params.get("limit");
        Page<SysUserEntity> pageTask =
                new Page<SysUserEntity>(Integer.parseInt(page), Integer.parseInt(limit));
        Page<SysUserEntity> page1 = sysUserService.queryByList(pageTask, model);
        return R.ok().put("page", page1);
    }

    /**
     * 获取登录的用户信息
     */
    @ApiOperation(value = "获取登录的用户信息", notes = "获取登录的用户信息")
    @GetMapping("/info")
    public R info() {
        return R.ok().put("user", getUser());
    }

    /**
     * 修改登录用户密码
     */
    @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "修改用户密码")
    @ApiOperation(value = "修改密码", notes = "修改密码")
    @ApiImplicitParam(
            name = "form",
            value = "密码表单实体",
            required = true,
            paramType = "query",
            dataType = "PasswordForm")
    @PostMapping("/password")
    public R password(@RequestBody PasswordForm form) {
        Assert.isBlank(form.getNewPassword(), "新密码不为能空");
        // 用户信息
        SysUserEntity user = sysUserService.queryByUserName(form.getUsername());

        // sha256加密
        String password = new Sha256Hash(form.getPassword(), user.getSalt()).toHex();
        // sha256加密
        String newPassword = new Sha256Hash(form.getNewPassword(), user.getSalt()).toHex();
        // 账号不存在、密码错误
        if (!user.getPassword().equals(password)) {
            return R.error("原密码不正确");
        } else {
            sysUserService.updatePassword(
                    form.getUsername(), newPassword, getUser().getSysSystem().getCode());
            return R.ok();
        }
        //		String password = new Sha256Hash(form.getPassword(), getUser().getSalt()).toHex();
        //		System.out.println(getUser().getCreateUserId());
        //		System.out.println(getUser().getSalt());

        // 更新密码
        //		boolean flag = sysUserService.updatePassword(getUserId(), password, newPassword);
        //		if(!flag){
        //			return R.error("原密码不正确");
        //		}

    }

    /**
     * 判断用户名是否重复
     */
    @GetMapping("/checkUser")
    @ApiOperation(value = "判断用户名是否重复", notes = "判断用户名是否重复")
    public R checkedUser(@RequestParam Map<String, Object> params) {
        SysUserEntity sysUser = new SysUserEntity();
        String username = (String) params.get("username");
        sysUser.setUsername(username);
        Integer num = sysUserService.queryUserById(sysUser);
        return R.ok().put("num", num);
    }

    /**
     * 通過用戶名獲取用户信息
     */
    @SneakyThrows
    @ApiOperation(value = "根据用户信息获取用户信息", notes = "根据用户信息获取用户信息")
    @ApiImplicitParam(
            name = "userName",
            value = "主键",
            required = true,
            paramType = "query",
            dataType = "String")
    @GetMapping("/infoByName/{userName}")
    public SysUserEntity infoByName(@PathVariable("userName") String userName, String stationId) {
        // 获取当前站点信息
        String u = Aes.decrypt(userName);
        SysUserEntity user = sysUserService.queryByUserName(u);

        // 获取用户所属的角色列表
        List<String> roleIdList = sysUserRoleService.queryRoleIdList(user.getUserId());
        user.setRoleIdList(roleIdList);

        return user;
    }

    /**
     * 保存用户
     */
    @SysLogMethod(operation = "新增", blockName = "系统管理", detail = "新增用户信息")
    @ApiOperation(value = "保存用户", notes = "保存用户")
    @ApiImplicitParam(
            name = "user",
            value = "用户实体",
            required = true,
            paramType = "query",
            dataType = "SysUserEntity")
    @PostMapping("/save")
    public R save(@RequestBody SysUserEntity user) {
        String idCard = user.getIdCard();
        List<SysUserEntity> list =
                sysUserService.list(new LambdaQueryWrapper<SysUserEntity>().eq(SysUserEntity::getIdCard, idCard).eq(SysUserEntity::getDeleted, 0));
        for (SysUserEntity sysUserEntity : list) {
            if (sysUserEntity.getIdCard().equals(idCard)) {
                return R.error("身份证信息已存在！");
            }
        }

        user.setCreateUserId(getUserId());
        user.setSysSign(getUser().getSysSystem().getCode());
        sysUserService.save(user);

        return R.ok();
    }

    /**
     * 修改用户
     */
    @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "修改用户信息")
    @ApiOperation(value = "修改用户", notes = "修改用户")
    @PostMapping("/update")
    public R update(@RequestBody @ApiParam(value = "用户实体", required = true) SysUserEntity user) {
        // 校验参数
        String idCard = user.getIdCard();
        List<SysUserEntity> list =
                sysUserService.list(
                        new QueryWrapper<SysUserEntity>()
                                .lambda()
                                .eq(SysUserEntity::getIdCard, idCard)
                                .eq(SysUserEntity::getDeleted, 0));
        for (SysUserEntity sysUserEntity : list) {
            if (!sysUserEntity.getUserId().equals(user.getUserId())
                    && sysUserEntity.getIdCard().equals(idCard)) {
                return R.error("身份证信息已存在！");
            }
        }
        user.setUpdateUserId(getUserId());
        user.setSysSign(getUser().getSysSystem().getCode());
        sysUserService.update(user);
        return R.ok();
    }

    /**
     * 删除用户
     */
    @SysLog("删除用户")
    @SysLogMethod(operation = "删除", blockName = "系统管理", detail = "删除用户信息")
    @ApiOperation(value = "删除用户", notes = "删除用户")
    @ApiImplicitParam(
            name = "userIds",
            value = "用户主键(多个用','隔开)",
            required = true,
            paramType = "query",
            dataType = "String")
    @PostMapping("/delete")
    public R delete(@RequestBody String[] userIds) {
        if (userIds == null || userIds.length == 0) {
            return R.ok();
        }
        if (ArrayUtils.contains(userIds, Constant.SUPER_ADMIN)) {
            return R.error("系统管理员不能删除");
        }

        if (ArrayUtils.contains(userIds, getUserId())) {
            return R.error("当前用户不能删除");
        }

        sysUserService.deleteBatch(userIds, getUser().getSysSystem().getCode());

        return R.ok();
    }

    /**
     * 根据参数查询所有用户
     */
    @SysLog("查询所有用户")
    @ApiOperation(value = "根据参数查询所有用户", notes = "根据参数查询所有用户")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "orgId", value = "部门", required = true, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "username", value = "账号", required = false, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "name", value = "姓名", required = false, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "jobNumber", value = "工号", required = false, paramType = "query", dataType = "String"),
            @ApiImplicitParam(name = "gender", value = "性别 0：女 1：男", required = false, paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "status", value = "状态0：禁用1：正常", required = false, paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "page", value = "当前页", required = false, paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "size", value = "每页显示条目数", required = false, paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "stationId", value = "站点id", required = false, paramType = "query", dataType = "String"),
    })
    @GetMapping("/selectUserByOrgId")
    public R selectUserByOrgId(String orgId, String username, String name, String jobNumber, Integer gender, String status, String sort, String order, Integer page, Integer size, String stationId) {
        return sysUserService.selectUserByOrgId(
                orgId, username, name, jobNumber, gender, status, sort, order, page, size, stationId);
    }

    /**
     * 用户信息（包含角色信息）
     */
    @ApiOperation(value = "根据ID获取用户信息", notes = "根据ID获取用户信息")
    @ApiImplicitParam(
            name = "userId",
            value = "主键",
            required = true,
            paramType = "query",
            dataType = "String")
    @GetMapping("/info/{userId}")
    public R getUserInfo(@PathVariable("userId") String userId) {
        SysUserEntity user = sysUserService.getById(userId);
        // 获取用户所属的角色列表
        List<SysRoleEntity> list = sysRoleService.queryUserRoleList(userId);
        if (!list.isEmpty()) {
            user.setUserRoleList(list);
        }
        return R.ok().put("user", user);
    }

    /**
     * 重置密码
     *
     * @param userIds
     * @param password
     * @return
     */
    @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "重置用户密码")
    @ApiOperation(value = "根据ID获取用户信息", notes = "根据ID获取用户信息")
    @ApiImplicitParam(
            name = "userIds",
            value = "用户ids(多个用户id用逗号隔开）",
            required = true,
            paramType = "query",
            dataType = "String")
    @PostMapping("/updatePassword")
    public R updatePassword(String userIds, String password) {
        if (StringUtils.isBlank(userIds)) {
            return R.error("请选择用户！");
        }
        String salt = RandomStringUtils.randomAlphanumeric(20);

        String[] split = userIds.split(",");
        String pd = new Sha256Hash(password, salt).toHex();
        sysUserService.updatePw(split, pd, salt, getUserId(), getUser().getSysSystem().getCode());
        return R.ok();
    }

    /** 获取民族 */
    @ApiOperation(value = "获取民族列表", notes = "获取民族列表")
    @GetMapping("/getAllNation")
    public R getAllNation() {
        List<HashMap<String, Object>> result = sysUserService.getAllNation();
        return R.ok().put("result", result);
    }
}
