package com.devplatform.admin.modules.sys.oauth2;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.devplatform.admin.common.utils.JwtUtil;
import com.devplatform.admin.modules.sys.bean.SysSystem;
import com.devplatform.admin.modules.sys.bean.SysUserEntity;
import com.devplatform.admin.modules.sys.service.ShiroService;
import com.devplatform.common.base.exception.RRException;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.SignatureException;
import java.util.Set;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationInfo;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.SimpleAuthenticationInfo;
import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.authz.SimpleAuthorizationInfo;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.subject.PrincipalCollection;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 认证
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2017-05-20 14:00
 */
@Component
public class Oauth2Realm extends AuthorizingRealm {
  @Autowired private ShiroService shiroService;

  @Override
  public boolean supports(AuthenticationToken token) {
    return token instanceof Oauth2Token;
  }

  /** 授权(验证权限时调用) */
  @Override
  protected AuthorizationInfo doGetAuthorizationInfo(PrincipalCollection principals) {
    SysUserEntity user = (SysUserEntity) principals.getPrimaryPrincipal();
    String userId = user.getUserId();

    // 用户权限列表
    Set<String> permsSet = shiroService.getUserPermissions(userId);

    SimpleAuthorizationInfo info = new SimpleAuthorizationInfo();
    info.setStringPermissions(permsSet);
    return info;
  }

  /** 认证(登录时调用) */
  @Override
  protected AuthenticationInfo doGetAuthenticationInfo(AuthenticationToken token)
      throws AuthenticationException {
    String accessToken = (String) token.getPrincipal();

    SysUserEntity user = new SysUserEntity();
    if (accessToken == null) {
      throw new RRException("缺少或无效的token");
    }
    final String jwtToken = accessToken;
    try {
      final Claims claims = JwtUtil.parseJwt(jwtToken);
      String subject = claims.getSubject();
      JSONObject json = JSON.parseObject(subject);
      String userId = json.getString("userId");
      user.setUserId(userId);
      String userName = json.getString("userName");
      user.setUsername(userName);

      String name = json.getString("name");
      user.setName(name);

      String orgId = json.getString("orgId");
      user.setOrgId(orgId);
      String system = json.getString("system");
      user.setSysSystem(JSON.parseObject(system, SysSystem.class));
    } catch (final SignatureException e) {
      throw new RRException("token解析异常");
    } catch (final ExpiredJwtException e) {
      throw new RRException("token过期，请重新登陆");
    }

    SimpleAuthenticationInfo info = new SimpleAuthenticationInfo(user, accessToken, getName());
    return info;
  }
}
