package com.devplatform.alarm.aspect;

import com.alibaba.fastjson.JSON;
import com.devplatform.alarm.common.annotation.SysLogMethod;
import com.devplatform.alarm.common.utils.AbstractController;
import com.devplatform.alarm.common.utils.Constants;
import com.devplatform.alarm.common.utils.HttpContextUtils;
import com.devplatform.alarm.common.utils.IpUtils;
import com.devplatform.alarm.feign.bean.SysUserEntity;
import com.devplatform.alarm.feign.service.LogService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 系统日志，切面处理类
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2017年3月8日 上午11:07:35
 */
@Aspect
@Component
public class SysLogAspect extends AbstractController {

  private static Log log = LogFactory.getLog(SysLogAspect.class);

  @Autowired(required = false)
  private LogService sysLogService;

  @Pointcut("@annotation(com.devplatform.alarm.common.annotation.SysLogMethod)")
  public void logPointCut() {
    log.info("");
  }

  @Around("logPointCut(),logMethodPointCut()")
  public Object around(ProceedingJoinPoint point) throws Throwable {
    long beginTime = System.currentTimeMillis();
    // 执行方法
    Object result = point.proceed();
    // 执行时长(毫秒)
    long time = System.currentTimeMillis() - beginTime;
    // 保存日志
    saveSysLog(point, time);
    return result;
  }

  private void saveSysLog(ProceedingJoinPoint joinPoint, long time) {
    MethodSignature signature = (MethodSignature) joinPoint.getSignature();
    Method method = signature.getMethod();
    Map<String, Object> params = new HashMap<String, Object>(8);
    SysLogMethod sysLogMethod = method.getAnnotation(SysLogMethod.class);
    if (sysLogMethod != null) {
      // 注解上的描述
      params.put("operation", sysLogMethod.operation());
      params.put("blockName", sysLogMethod.blockName());
      params.put("detail", sysLogMethod.detail());
    }
    // 请求的方法名
    String className = joinPoint.getTarget().getClass().getName();
    String methodName = signature.getName();
    params.put("method", className + "." + methodName + "()");

    // 请求的参数
    Object[] args = joinPoint.getArgs();

    StringBuilder requestBuilder = new StringBuilder(0);

    try {
      for (int index = 0; index < args.length; index++)
      {
        // 如果参数类型是请求和响应的http，则不需要拼接【这两个参数，使用JSON.toJSONString()转换会抛异常】
        if (args[index] instanceof HttpServletRequest
                || args[index] instanceof HttpServletResponse)
        {
          continue;
        }
        requestBuilder.append(args[index] == null ? "" : JSON.toJSONString(args[index]));
      }
      String jsons = requestBuilder.toString();
      params.put("params", jsons);
    } catch (Exception e) {
      e.printStackTrace();
    }
    // 获取request
    HttpServletRequest request = HttpContextUtils.getHttpServletRequest();
    SysUserEntity user = getUser();
    String stationId = request.getHeader("stationId");
    if (Constants.STRING_1.equals(user.getSysSystem().getType()) && !Constants.CHECK.equals(sysLogMethod.operation())) {
      params.put("stationId", user.getSysSystem().getCode());
      params.put("name", user.getName());
    } else if (Constants.STRING_2.equals(user.getSysSystem().getType())) {
      params.put("stationId", stationId);
      params.put("name", "线路管理员");
    } else if (Constants.STRING_3.equals(user.getSysSystem().getType())) {
      params.put("stationId", stationId);
      params.put("name", "路网管理员");
    } else if (Constants.STRING_1.equals(user.getSysSystem().getType()) && Constants.CHECK.equals(sysLogMethod.operation())) {
      // 站点自己访问时不记录查看日志
      return;
    }
    // 设置IP地址
    params.put("ip", IpUtils.getIpAddr(request));
    params.put("time", time);
    params.put("createDate", new Date());
    params.put("createBy", getUserId());
    // 保存系统日志
    sysLogService.saveSysLog(params);
  }
}
