package com.devplatform.alarm.common.producer;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.devplatform.alarm.common.utils.Constants;
import com.devplatform.alarm.modules.synclog.bean.SyncLogData;
import com.devplatform.alarm.modules.synclog.service.SyncLogDataService;
import com.devplatform.routes.modules.plan.bean.RtPlanRegister;
import lombok.SneakyThrows;
import org.apache.commons.lang.StringUtils;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.rabbit.support.CorrelationData;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.stereotype.Component;

import java.io.ByteArrayInputStream;
import java.io.ObjectInputStream;
import java.util.Date;
import java.util.Map;
import java.util.UUID;

/**
 * 线路预案下发生产者
 * @Author: jzj
 * @Date: 2020/9/25 13:38
 */
@Component
public class PlanRabbitSender {

    @Autowired
    private RabbitTemplate rabbitTemplate;

    @Autowired
    private SyncLogDataService syncLogDataService;

    /**
     * 消息确认机制
     */
    final RabbitTemplate.ConfirmCallback confirmCallback = new RabbitTemplate.ConfirmCallback(){
        @Override
        public void confirm(CorrelationData correlationData, boolean ack, String s) {
//            System.err.println("correlationData:" + correlationData.toString());
//            System.err.println("ack:" + ack);
//            System.err.println("消息确认发送成功"+s);
            //消息未发送成功，则同步失败
            if(!ack){
                //通过消息id查询日志
                SyncLogData syncLogData = syncLogDataService.getOne(new LambdaQueryWrapper<SyncLogData>().eq(SyncLogData::getRemark,correlationData.getId())
                        .eq(SyncLogData::getType, Constants.INT_1));
                if(syncLogData != null){
                    syncLogData.setStatus(2);
                    syncLogDataService.update(syncLogData,new LambdaQueryWrapper<SyncLogData>().eq(SyncLogData::getId,syncLogData.getId()));
                }
            }
        }
    };

    /**
     *当我们发送消息时，当前exchange不存在或者指定的路由Key路由不到，这个时候我们需要监听这个种不可达的消息
     */
    final RabbitTemplate.ReturnCallback returnCallback = new RabbitTemplate.ReturnCallback() {
        @SneakyThrows
        @Override
        public void  returnedMessage(org.springframework.amqp.core.Message message, int i, String s, String exchange, String routingKey) {
            Message msg = byteToObject(message.getBody(), Message.class);
            MessageHeaders mhs = msg.getHeaders();
            String type = (String) mhs.get("type");
            Thread.sleep(500);
            //同步失败
            if(Constants.STRING_1.equals(type)){
                RtPlanRegister bean = (RtPlanRegister) msg.getPayload();
                createLog(bean.getId(),bean.getStationId(),2);
            }
        }
    };

    /**
     * 发送消息方法调用: 构建Message消息
     * @param message
     * @param properties
     * @param exchange
     * @param routingKey
     * @throws Exception
     */
    public void send(Object message, Map<String, Object> properties,String exchange,String routingKey) throws Exception {
        MessageHeaders mhs = new MessageHeaders(properties);
        Message msg = MessageBuilder.createMessage(message, mhs);
        rabbitTemplate.setConfirmCallback(confirmCallback);
        rabbitTemplate.setReturnCallback(returnCallback);
        CorrelationData correlationData = new CorrelationData();
        correlationData.setId(UUID.randomUUID().toString());
        rabbitTemplate.convertAndSend(exchange, routingKey, msg,correlationData);
    }

    /**
     * 发送消息方法调用
     * @param planRegister
     * @param properties
     * @param exchange
     * @param routingKey
     * @throws Exception
     */
    public void sendPlanRegister(RtPlanRegister planRegister, Map<String, Object> properties, String exchange, String routingKey) throws Exception {
        MessageHeaders mhs = new MessageHeaders(properties);
        Message msg = MessageBuilder.createMessage(planRegister, mhs);
        rabbitTemplate.setReturnCallback(returnCallback);
        rabbitTemplate.setConfirmCallback(confirmCallback);
        CorrelationData correlationData = new CorrelationData(UUID.randomUUID().toString());
        rabbitTemplate.convertAndSend(exchange,routingKey,msg,correlationData);

        //生成状态为同步中的日志
        createNewLog(planRegister.getId(),planRegister.getStationId(),3,correlationData.getId());
    }

    /**
     * 默认创建同步中的日志记录
     * @param planId
     * @param stationId
     */
    private void createNewLog(String planId,String stationId,Integer status,String msgId){
        SyncLogData syncLogData = syncLogDataService.getOne(new LambdaQueryWrapper<SyncLogData>().eq(SyncLogData::getPlanId,planId)
                .eq(SyncLogData::getStationId,stationId).eq(SyncLogData::getType, Constants.INT_1));
        if(syncLogData == null){
            syncLogData = new SyncLogData();
            syncLogData.setPlanId(planId);
            syncLogData.setStationId(stationId);
            syncLogData.setCreateTime(new Date());
            syncLogData.setType(1);
            syncLogData.setStatus(status);
            syncLogData.setRemark(msgId);
            syncLogDataService.save(syncLogData);
        }else {
            syncLogData.setStatus(status);
            syncLogData.setRemark(msgId);
            syncLogDataService.update(syncLogData,new LambdaQueryWrapper<SyncLogData>().eq(SyncLogData::getId,syncLogData.getId()));
        }
    }

    /**
     * 默认创建失败的日志记录
     * @param planId
     * @param stationId
     */
    private void createLog(String planId,String stationId,Integer status){
        SyncLogData syncLogData = syncLogDataService.getOne(new LambdaQueryWrapper<SyncLogData>().eq(SyncLogData::getPlanId,planId)
                .eq(SyncLogData::getStationId,stationId).eq(SyncLogData::getType, Constants.INT_1));
        if(syncLogData != null){
            syncLogData.setStatus(status);
            syncLogDataService.update(syncLogData,new LambdaQueryWrapper<SyncLogData>().eq(SyncLogData::getId,syncLogData.getId()));
        }
    }

    @SuppressWarnings("unchecked")
    private <T> T byteToObject(byte[] bytes, Class<T> clazz) {
        T t;
        try (ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
             ObjectInputStream ois = new ObjectInputStream(bis)) {
            t = (T) ois.readObject();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        return t;
    }

}
