package com.oauth.security.handler;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.sun.jersey.core.util.Base64;
import java.io.IOException;
import java.util.HashMap;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.common.exceptions.UnapprovedClientAuthenticationException;
import org.springframework.security.oauth2.provider.ClientDetails;
import org.springframework.security.oauth2.provider.ClientDetailsService;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.OAuth2Request;
import org.springframework.security.oauth2.provider.TokenRequest;
import org.springframework.security.oauth2.provider.token.AuthorizationServerTokenServices;
import org.springframework.security.web.authentication.SavedRequestAwareAuthenticationSuccessHandler;
import org.springframework.stereotype.Component;

/**
 * APP登录成功处理器 Created on 2018/7/21 0021.
 *
 * @author szy
 * @email i@suntray.com
 * @since 1.0
 */
@Slf4j
@Component("appLoginInSuccessHandler")
public class AppLoginInSuccessHandler extends SavedRequestAwareAuthenticationSuccessHandler {

  @Autowired private ObjectMapper objectMapper;

  @Autowired private PasswordEncoder passwordEncoder;

  @Autowired private ClientDetailsService clientDetailsService;

  @Autowired private AuthorizationServerTokenServices authorizationServerTokenServices;

  @Override
  public void onAuthenticationSuccess(
      HttpServletRequest request, HttpServletResponse response, Authentication authentication)
      throws ServletException, IOException {

    //        log.info("【AppLoginInSuccessHandler】 onAuthenticationSuccess authentication={}",
    // authentication);

    String header = request.getHeader("Authorization");
    String basic = "Basic ";
    if (header == null || !header.startsWith(basic)) {
      throw new UnapprovedClientAuthenticationException("请求头中无client信息");
    }
    String[] tokens = this.extractAndDecodeHeader(header, request);

    assert tokens.length == 2;

    String clientId = tokens[0];
    String clientSecret = tokens[1];

    ClientDetails clientDetails = clientDetailsService.loadClientByClientId(clientId);

    if (clientDetails == null) {
      throw new UnapprovedClientAuthenticationException("clientId 对应的配置信息不存在" + clientId);
    } else if (!passwordEncoder.matches(clientSecret, clientDetails.getClientSecret())) {
      throw new UnapprovedClientAuthenticationException("clientSecret 不匹配" + clientId);
    }

    TokenRequest tokenRequest =
        new TokenRequest(new HashMap<>(10), clientId, clientDetails.getScope(), "custom");

    OAuth2Request oAuth2Request = tokenRequest.createOAuth2Request(clientDetails);

    OAuth2Authentication oAuth2Authentication =
        new OAuth2Authentication(oAuth2Request, authentication);

    OAuth2AccessToken token =
        authorizationServerTokenServices.createAccessToken(oAuth2Authentication);

    response.setContentType("application/json;charset=UTF-8");
    response.getWriter().write(objectMapper.writeValueAsString(token));
    //        log.info("token={}", JsonUtil.toJson(token));

  }

  /**
   * 解码
   *
   * @param header
   * @param request
   * @return
   * @throws IOException
   */
  private String[] extractAndDecodeHeader(String header, HttpServletRequest request)
      throws IOException {
    byte[] base64Token = header.substring(6).getBytes("UTF-8");

    byte[] decoded;
    try {
      decoded = Base64.decode(base64Token);
    } catch (IllegalArgumentException var7) {
      throw new BadCredentialsException("Failed to decode basic authentication token");
    }

    String token = new String(decoded, "UTF-8");
    int delim = token.indexOf(":");
    if (delim == -1) {
      throw new BadCredentialsException("Invalid basic authentication token");
    } else {
      return new String[] {token.substring(0, delim), token.substring(delim + 1)};
    }
  }
}
