package com.devplatform.equipment.aspect;

import com.alibaba.fastjson.JSON;
import com.devplatform.equipment.common.annotation.SysLogMethod;
import com.devplatform.equipment.common.utils.AbstractController;
import com.devplatform.equipment.common.utils.Constants;
import com.devplatform.equipment.common.utils.HttpContextUtils;
import com.devplatform.equipment.common.utils.IpUtils;
import com.devplatform.equipment.feign.bean.SysUserEntity;
import com.devplatform.equipment.feign.service.LogService;

import java.lang.reflect.Method;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 系统日志，切面处理类
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2017年3月8日 上午11:07:35
 */
@Aspect
@Component
public class SysLogAspect extends AbstractController {

  private static Log log = LogFactory.getLog(SysLogAspect.class);

  @Autowired(required = false)
  private LogService logService;

  @Pointcut("@annotation(com.devplatform.equipment.common.annotation.SysLogMethod)")
  public void logPointCut() {
    log.info("");
  }

  @Around("logPointCut(),logMethodPointCut()")
  public Object around(ProceedingJoinPoint point) throws Throwable {
    long beginTime = System.currentTimeMillis();
    // 执行方法
    Object result = point.proceed();
    // 执行时长(毫秒)
    long time = System.currentTimeMillis() - beginTime;
    // 保存日志
    saveSysLog(point, time);
    return result;
  }

  private void saveSysLog(ProceedingJoinPoint joinPoint, long time) {
    MethodSignature signature = (MethodSignature) joinPoint.getSignature();
    Method method = signature.getMethod();
    Map<String, Object> params = new HashMap<String, Object>(8);
    SysLogMethod sysLogMethod = method.getAnnotation(SysLogMethod.class);
    if (sysLogMethod != null) {
      // 注解上的描述
      params.put("operation", sysLogMethod.operation());
      params.put("blockName", sysLogMethod.blockName());
      params.put("detail", sysLogMethod.detail());
    }
    // 请求的方法名
    String className = joinPoint.getTarget().getClass().getName();
    String methodName = signature.getName();
    params.put("method", className + "." + methodName + "()");

    // 请求的参数
    Object[] args = joinPoint.getArgs();
    try {
      String jsons = JSON.toJSONString(args);
      params.put("params", jsons);
    } catch (Exception e) {
      e.printStackTrace();
    }
    // 获取request
    HttpServletRequest request = HttpContextUtils.getHttpServletRequest();
    SysUserEntity user = getUser();
    String stationId = request.getHeader("stationId");
    if (Constants.STRING_1.equals(user.getSysSystem().getType()) && !Constants.CHECK.equals(sysLogMethod.operation())) {
      params.put("stationId", user.getSysSystem().getCode());
      params.put("name", user.getName());
    } else if (Constants.STRING_2.equals(user.getSysSystem().getType())) {
      params.put("stationId", stationId);
      params.put("name", "线路管理员");
    } else if (Constants.STRING_3.equals(user.getSysSystem().getType())) {
      params.put("stationId", stationId);
      params.put("name", "路网管理员");
    } else if (Constants.STRING_1.equals(user.getSysSystem().getType()) && Constants.CHECK.equals(sysLogMethod.operation())) {
      // 站点自己访问时不记录查看日志
      return;
    }
    // 设置IP地址
    params.put("ip", IpUtils.getIpAddr(request));
    params.put("time", time);
    params.put("createDate", new Date());
    params.put("createBy", getUserId());
    // 保存系统日志
    logService.saveSysLog(params);
  }
}
