package com.devplatform.equipment.modules.eppatrolline.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.devplatform.common.util.R;
import com.devplatform.common.util.StringUtil;
import com.devplatform.equipment.common.annotation.SysLogMethod;
import com.devplatform.equipment.common.utils.AbstractController;
import com.devplatform.equipment.modules.eppatrolline.bean.EpPatrolLine;
import com.devplatform.equipment.modules.eppatrolline.model.EpPatrolLineModel;
import com.devplatform.equipment.modules.eppatrolline.service.EpPatrolLineService;
import com.devplatform.equipment.modules.eppatrollinepoint.bean.EpPatrolLinePoint;
import com.devplatform.equipment.modules.eppatrollinepoint.service.EpPatrolLinePointService;
import com.devplatform.equipment.modules.eppatrolplan.service.EpPatrolPlanService;
import com.devplatform.equipment.modules.listationmap.bean.LiStationMap;
import com.devplatform.equipment.modules.listationmap.service.LiStationMapService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * @author Administrator
 */
@Api(tags = {"电子巡查线路表管理接口"})
@RestController
@RequestMapping("/epPatrolLine")
public class EpPatrolLineController extends AbstractController {

  @Autowired private EpPatrolLineService epPatrolLineService;

  @Autowired private EpPatrolPlanService epPatrolPlanService;

  @Autowired private EpPatrolLinePointService epPatrolLinePointService;

  @Autowired private LiStationMapService liStationMapService;

  /**
   * 列表页面列表数据获取
   *
   * @param params 参数 stationId 站点id
   * @return
   */
  @ApiOperation(value = "获取电子巡查线路表管理数据", notes = "根据条件获取电子巡查线路表管理数据")
  @ApiImplicitParam(name = "params", value = "参数", required = true, dataType = "Map<String,String>")
  @PostMapping("/list")
  public R list(@RequestBody Map<String, String> params) {
    List<Map<String, Object>> nodes = new ArrayList<>();
    String stationId = params.get("stationId");
    // 根据站点id找巡查线路
    List<EpPatrolLine> epPatrolLineList = epPatrolLineService.getTreeListByStationId(stationId);
    if (epPatrolLineList != null && epPatrolLineList.size() > 0) {
      for (int i = 0; i < epPatrolLineList.size(); i++) {
        Map<String, Object> map = new LinkedHashMap<String, Object>();
        map.put("id", epPatrolLineList.get(i).getId());
	// 巡查线路名称
        map.put("name", epPatrolLineList.get(i).getName());
	// 巡查线路描述i
        map.put("sysSign", epPatrolLineList.get(i).getSysSign());
	// 关联计划的个数
        map.put("count", epPatrolLineList.get(i).getCount());
	// 关联计划的个数
        map.put("byx1", epPatrolLineList.get(i).getByx1());
        List<String> optionIds = new ArrayList<>();
        List<double[]> coordinateList = new ArrayList<>();
        // 根据巡查线路查找巡查线路下巡查点
        List<EpPatrolLinePoint> epPatrolLinePointList =
            epPatrolLinePointService.getTreeListByLineId(
                epPatrolLineList.get(i).getId(), stationId);
        if (epPatrolLinePointList != null && epPatrolLinePointList.size() > 0) {
          List<Map<String, Object>> children = new ArrayList<>();
          for (int j = 0; j < epPatrolLinePointList.size(); j++) {
            Map<String, Object> mapChildren = new LinkedHashMap<>();
            map.put("layerId", epPatrolLinePointList.get(j).getLayerId());
	// 巡查点id
            mapChildren.put("id", epPatrolLinePointList.get(j).getPointId());
	// 巡查点名
            mapChildren.put("name", epPatrolLinePointList.get(j).getName());
	// 站层id
            mapChildren.put("layerId", epPatrolLinePointList.get(j).getLayerId());
            mapChildren.put("byx2", epPatrolLinePointList.get(j).getPointByx2());
            double[] coordinate = new double[2];
            if (StringUtils.isNotBlank(epPatrolLinePointList.get(j).getCoordinateX())) {
              coordinate[0] = Double.parseDouble(epPatrolLinePointList.get(j).getCoordinateX());
            }
            if (StringUtils.isNotBlank(epPatrolLinePointList.get(j).getCoordinateY())) {
              coordinate[1] = Double.parseDouble(epPatrolLinePointList.get(j).getCoordinateY());
            }
            mapChildren.put("coordinateY", epPatrolLinePointList.get(j).getCoordinateY());
            mapChildren.put("coordinateX", epPatrolLinePointList.get(j).getCoordinateX());
            children.add(mapChildren);
            coordinateList.add(coordinate);
            optionIds.add(epPatrolLinePointList.get(j).getPointId());
          }
          map.put("children", children);
          map.put("coordinate", coordinateList);
          map.put("optionIds", optionIds);
        }
        nodes.add(map);
      }
    }
    return R.ok().put("nodes", nodes);
  }

  /**
   * 添加
   *
   * @param bean 电子巡查线路对象 stationId 站点id name 巡查线路名称 sysSign 描述
   * @return
   */
  @SysLogMethod(operation = "新增", blockName = "电子巡更", detail = "新增电子巡查线路数据")
  @ApiOperation(value = "新增电子巡查线路表管理数据", notes = "新增电子巡查线路表管理数据")
  @PostMapping("/save")
  public R save(
      @RequestBody @ApiParam(value = "电子巡查线路表管理实体对象", required = true) EpPatrolLine bean) {
    bean.setCreateUserId(getUserId());
    bean.setCreateTime(new Date());
    bean.setSysSign(getUser().getSysSystem().getCode());
    epPatrolLineService.save(bean);
    return R.ok();
  }

  /**
   * 修改
   *
   * @param bean 电子巡查线路对象
   * @return
   */
  @SysLogMethod(operation = "修改", blockName = "电子巡更", detail = "修改电子巡查线路数据")
  @ApiOperation(value = "修改电子巡查线路表管理数据", notes = "修改电子巡查线路表管理数据")
  @PostMapping("/update")
  public R update(
      @RequestBody @ApiParam(value = "电子巡查线路表管理实体对象", required = true) EpPatrolLine bean) {
    if (StringUtil.checkNotNull(bean.getId())) {
      // 判断此巡查线路有没有被加入巡查计划，有则不能修改 ，且给提示信息
      if (epPatrolPlanService.countByLineId(bean.getId(), bean.getStationId()) <= 0) {
        bean.setUpdateUserId(getUserId());
        bean.setUpdateTime(new Date());
        bean.setSysSign(getUser().getSysSystem().getCode());
        epPatrolLineService.updateById(bean);
      } else {
        return R.error("此巡查线路已被加入巡查计划，不能修改");
      }
    }
    return R.ok();
  }

  /**
   * 判断此巡查线路有没有被加入巡查计划
   *
   * @param bean 电子巡查线路对象
   * @return
   */
  @ApiOperation(value = "判断此巡查线路有没有被加入巡查计划", notes = "判断此巡查线路有没有被加入巡查计划")
  @PostMapping("/checkLine")
  public R checkLine(
      @RequestBody @ApiParam(value = "电子巡查线路表管理实体对象", required = true) EpPatrolLine bean) {
    // 判断此巡查线路有没有被加入巡查计划，有则不能修改 ，且给提示信息
    if (epPatrolPlanService.countByLineId(bean.getId(), bean.getStationId()) > 0) {
      return R.error("此巡查线路已被加入巡查计划，不能修改");
    }
    return R.ok();
  }

  @GetMapping("/getCountByLineId")
  public R getCountByLineId(String id, String stationId) {
    Integer count = epPatrolPlanService.countByLineId(id, stationId);
    if (count <= 0) {
      return R.ok();
    } else {
      return R.error("此巡查线路已被加入巡查计划，不能修改其巡查点信息");
    }
  }

  /**
   * 保存巡查线路的巡查点信息
   *
   * @param model 承接对象 id 巡查线路主键 list 巡查点相关数据集合{ layerId 巡查点站层id id 巡查点id集合 name 巡查点名字
   *     <p>}
   * @return
   */
  @SysLogMethod(operation = "修改", blockName = "电子巡更", detail = "修改电子巡查线路下巡查点")
  @ApiOperation(value = "保存电子巡查线路下巡查点", notes = "保存电子巡查线路下巡查点")
  @PostMapping("/savePoint")
  public R savePoint(
      @RequestBody @ApiParam(value = "电子巡查线路表管理实体对象", required = true) EpPatrolLineModel model) {
    if (StringUtil.checkNotNull(model.getId())) {
      // 判断此巡查线路有没有被加入巡查计划，有则不能保存巡查点信息 ，且给提示信息
      if (epPatrolPlanService.countByLineId(model.getId(), model.getStationId()) <= 0) {
        epPatrolLineService.savePoint(model, getUserId(), getUser().getSysSystem().getCode());
      } else {
        return R.error("此巡查线路已被加入巡查计划，不能修改其巡查点信息");
      }
    }

    return R.ok();
  }

  /**
   * 根据ID获取电子巡查线路表管理对象
   *
   * @param bean 电子巡查线路表管理对象 id 巡查路线主键
   * @return
   */
  @SysLogMethod(operation = "删除", blockName = "电子巡更", detail = "删除电子巡查线路数据")
  @ApiOperation(value = "根据ID删除电子巡查线路表管理数据", notes = "根据ID删除电子巡查线路表管理数据")
  @PostMapping("/delete")
  public R delete(@RequestBody EpPatrolLine bean) {
    if (StringUtil.checkNotNull(bean.getId())) {
      /// 判断此巡查线路有没有被加入巡查计划，有则不能删除 ，且给提示信息
      if (epPatrolPlanService.countByLineId(bean.getId(), bean.getStationId()) <= 0) {
        epPatrolLineService.deleteById(
            bean.getId(), bean.getStationId(), getUser().getSysSystem().getCode());
      } else {
        return R.error("此巡查线路已被加入巡查计划，不能删除");
      }
    }
    return R.ok();
  }

  /**
   * 根据巡查线路ID查找巡查线路下巡查点
   *
   * @param lineId 对象主键
   * @return
   */
  @ApiOperation(value = "根据巡查线路ID查找巡查线路下巡查点", notes = "根据巡查线路ID查找巡查线路下巡查点")
  @ApiImplicitParam(name = "lineId", required = true, dataType = "String")
  @GetMapping("/getByLineId")
  public R getId(String lineId, String stationId) {
    List<EpPatrolLinePoint> epPatrolLinePointList =
        epPatrolLinePointService.getTreeListByLineId(lineId, stationId);
    // 查询当前站点相应所有层级
    LiStationMap map =
        liStationMapService.getOne(
            new LambdaQueryWrapper<LiStationMap>()
                .eq(LiStationMap::getId, epPatrolLinePointList.get(0).getLayerId())
                .eq(
                    StringUtil.checkNotNull(epPatrolLinePointList.get(0).getStationId()),
                    LiStationMap::getStationId,
                    epPatrolLinePointList.get(0).getStationId()));
    return R.ok().put("list", epPatrolLinePointList).put("map", map);
  }

  /**
   * 根据巡查线路ID查找巡查线路下巡查点
   *
   * @param lineId 对象主键
   * @return
   */
  @ApiOperation(value = "根据巡查线路ID查找巡查线路下巡查点", notes = "根据巡查线路ID查找巡查线路下巡查点")
  @ApiImplicitParam(name = "lineId", required = true, dataType = "String")
  @GetMapping("/getResultByLineId")
  public R getResultByLineId(String lineId, String stationId, String planId) {
    List<EpPatrolLinePoint> epPatrolLinePointList =
        epPatrolLinePointService.getResultByLineId(lineId, stationId,planId);
    // 查询当前站点相应所有层级
    LiStationMap map =
        liStationMapService.getOne(
            new LambdaQueryWrapper<LiStationMap>()
                .eq(LiStationMap::getId, epPatrolLinePointList.get(0).getLayerId())
                .eq(
                    StringUtil.checkNotNull(epPatrolLinePointList.get(0).getStationId()),
                    LiStationMap::getStationId,
                    epPatrolLinePointList.get(0).getStationId()));
    return R.ok().put("list", epPatrolLinePointList).put("map", map);
  }
}
