package com.devplatform.equipment.modules.eppatrolmachine.controller;

import com.devplatform.common.base.validator.ValidatorUtils;
import com.devplatform.common.util.R;
import com.devplatform.common.util.StringUtil;
import com.devplatform.equipment.common.annotation.SysLogMethod;
import com.devplatform.equipment.common.utils.AbstractController;
import com.devplatform.equipment.common.utils.Constants;
import com.devplatform.equipment.common.utils.MyPage;
import com.devplatform.equipment.modules.eppatrolmachine.bean.EpPatrolMachine;
import com.devplatform.equipment.modules.eppatrolmachine.model.EpPatrolMachineModel;
import com.devplatform.equipment.modules.eppatrolmachine.service.EpPatrolMachineService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import sun.misc.BASE64Encoder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * @author Administrator
 */
@Api(tags = {"电子巡查巡查机表接口"})
@RestController
@RequestMapping("/epPatrolMachine")
public class EpPatrolMachineController extends AbstractController {

    @Autowired
    private EpPatrolMachineService epPatrolMachineService;

    /**
     * 列表页面列表数据获取
     *
     * @param model 承接对象 stationId 站点id keywords 关键字
     * @return
     */
    @ApiOperation(value = "根据条件获取电子巡查巡查机表分页数据列表", notes = "根据条件获取电子巡查巡查机表分页数据列表")
    @ApiImplicitParam(
            name = "model",
            value = "参数",
            required = true,
            dataType = "EpPatrolMachineModel")
    @PostMapping("/list")
    public R list(@RequestBody EpPatrolMachineModel model) {
        if (StringUtils.isBlank(model.getSort())) {
            model.setSort("create_time");
            model.setOrder("DESC");
        }
        List<EpPatrolMachine> list = epPatrolMachineService.queryPageByList(model);
        return R.ok().put("page", getMyPage(list, model));
    }

    /**
     * 添加
     *
     * @param bean 电子巡查巡更机表对象 stationId 站点id orgName 所属部门名称 name 巡更机名称 code 巡更机编码 startTime 启动时间
     * @return
     */
    @SysLogMethod(operation = "新增", blockName = "电子巡更", detail = "新增电子巡查巡查机数据")
    @ApiOperation(value = "新增电子巡查巡查机表数据", notes = "新增电子巡查巡查机表数据")
    @PostMapping("/save")
    public R save(
            @RequestBody @ApiParam(value = "电子巡查巡更机表实体对象", required = true) EpPatrolMachine bean) {
        ValidatorUtils.validateEntity(bean);
        bean.setCreateUserId(getUserId());
        bean.setCreateTime(new Date());
        bean.setSysSign(getUser().getSysSystem().getCode());
        epPatrolMachineService.save(bean);
        return R.ok();
    }

    /**
     * 修改
     *
     * @param bean 电子巡查巡更机表对象
     * @return
     */
    @SysLogMethod(operation = "修改", blockName = "电子巡更", detail = "修改电子巡查巡查机数据")
    @ApiOperation(value = "修改电子巡查巡查机表数据", notes = "修改电子巡查巡查机表数据")
    @PostMapping("/update")
    public R update(
            @RequestBody @ApiParam(value = "电子巡查巡查机表实体对象", required = true) EpPatrolMachine bean) {
        ValidatorUtils.validateEntity(bean);
        bean.setUpdateUserId(getUserId());
        bean.setUpdateTime(new Date());
        bean.setSysSign(getUser().getSysSystem().getCode());
        epPatrolMachineService.updateById(bean);
        return R.ok();
    }

    /**
     * 根据Id删除电子巡查巡更机表对象
     *
     * @param id 电子巡查巡更机表对象主键
     * @return
     */
    @SysLogMethod(operation = "删除", blockName = "电子巡更", detail = "删除电子巡查巡查机数据")
    @ApiOperation(value = "根据ID删除电子巡查巡查机表数据", notes = "根据ID删除电子巡查巡查机表数据")
    @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
    @GetMapping("/delete")
    public R delete(String id, String stationId) {
        if (StringUtil.checkNotNull(id)) {
            // 逻辑删除
            epPatrolMachineService.deleteById(id, stationId, getUser().getSysSystem().getCode());
        }
        return R.ok();
    }

    /**
     * 根据条件导出巡更机管理excel文件
     *
     * @param keywords  可能是巡更机名称 编码 所属部门
     * @param stationId 站点id
     */
    @ApiOperation(value = "根据条件导出巡查机管理excel文件", notes = "根据条件导出巡查机管理excel文件")
    @GetMapping("/export")
    public void export(String keywords, String stationId, HttpServletRequest request, HttpServletResponse rep) {
        EpPatrolMachineModel model = new EpPatrolMachineModel();
        model.setStationId(stationId);
        model.setKeywords(keywords);
        // 获取到需要导出的数据
        List<EpPatrolMachine> epPatrolMachineList = (List<EpPatrolMachine>) ((MyPage) list(model).get("page")).getRows();

        OutputStream os = null;
        try {
            /** 取得输出流 */
            os = rep.getOutputStream();
            /** 清空输出流 */
            rep.reset();
            rep.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
            String fileName = "巡查机管理.xlsx";
            String userAgent = request.getHeader("USER-AGENT");
            if (userAgent.contains(Constants.FIREFOX)) {
                // 是火狐浏览器，使用BASE64编码
                fileName = "=?utf-8?b?" + new BASE64Encoder().encode(fileName.getBytes("utf-8")) + "?=";
            } else {
                // 给文件名进行URL编码
                // URLEncoder.encode()需要两个参数，第一个参数时要编码的字符串，第二个是编码所采用的字符集
                fileName = URLEncoder.encode(fileName, "utf-8");
            }
            rep.setHeader("Content-disposition", "attachment;fileName=" + fileName);
            /** 设定输出文件头,输出类型 */
            rep.setContentType("application/msexcel");
            DateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            XSSFWorkbook workbook = new XSSFWorkbook();
            // 创建单元格样式
            XSSFCellStyle cellStyle = workbook.createCellStyle();
            cellStyle.setAlignment(HorizontalAlignment.CENTER);
            XSSFSheet sheet = workbook.createSheet("1");
            XSSFRow firstRow = sheet.createRow(0);
            firstRow.createCell(0).setCellValue("序号");
            firstRow.createCell(1).setCellValue("巡查机名称");
            firstRow.createCell(2).setCellValue("巡查机编码");
            firstRow.createCell(3).setCellValue("所属部门");
            firstRow.createCell(4).setCellValue("启用时间");
            firstRow.createCell(5).setCellValue("巡查次数");
            firstRow.createCell(6).setCellValue("使用总时长(分)");
            int index = 1;
            //设置单元格值和样式
            setCellValueAndStyle(epPatrolMachineList, sdf, cellStyle, sheet, firstRow, index);
            workbook.write(os);
            os.flush();

        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (os != null) {
                try {
                    os.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * 设置单元格值和样式
     *
     * @param epPatrolMachineList
     * @param sdf
     * @param cellStyle
     * @param sheet
     * @param firstRow
     * @param index
     */
    private void setCellValueAndStyle(List<EpPatrolMachine> epPatrolMachineList, DateFormat sdf, XSSFCellStyle cellStyle, XSSFSheet sheet, XSSFRow firstRow, int index) {
        for (EpPatrolMachine epPatrolMachine : epPatrolMachineList) {
            XSSFRow row = sheet.createRow(index);
            row.createCell(0).setCellValue(index);
            row.createCell(1).setCellValue(epPatrolMachine.getName());
            row.createCell(2).setCellValue(epPatrolMachine.getCode());
            row.createCell(3).setCellValue(epPatrolMachine.getOrgName());
            row.createCell(4).setCellValue(sdf.format(epPatrolMachine.getStartTime()));
            if (epPatrolMachine.getPatrolTimes() == null) {
                row.createCell(5).setCellValue("");
            } else {
                row.createCell(5).setCellValue(epPatrolMachine.getPatrolTimes());
            }
            if (epPatrolMachine.getDuration() == null) {
                row.createCell(6).setCellValue("");
            } else {
                row.createCell(6).setCellValue(epPatrolMachine.getDuration());
            }
            index++;
        }
        // 设置单元格样式
        for (int i = 0; i < sheet.getPhysicalNumberOfRows(); i++) {
            XSSFRow row = sheet.getRow(i);
            for (int j = 0; j < row.getPhysicalNumberOfCells(); j++) {
                if (row.getCell(j) == null) {
                    row.createCell(j).setCellStyle(cellStyle);
                } else {
                    row.getCell(j).setCellStyle(cellStyle);
                }
            }
        }
        for (int i = 0; i < firstRow.getPhysicalNumberOfCells(); i++) {
            // 设置列宽自适应
            sheet.autoSizeColumn(i);
            sheet.setColumnWidth(i, sheet.getColumnWidth(i) * 17 / 10);
        }
    }
}
