package com.devplatform.equipment.modules.eppatrolplan.controller;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.devplatform.common.util.R;
import com.devplatform.common.util.StringUtil;
import com.devplatform.equipment.common.annotation.SysLogMethod;
import com.devplatform.equipment.common.utils.AbstractController;
import com.devplatform.equipment.feign.bean.SysUserEntity;
import com.devplatform.equipment.modules.eppatrolplan.bean.EpPatrolPlan;
import com.devplatform.equipment.modules.eppatrolplan.bean.SendMessage;
import com.devplatform.equipment.modules.eppatrolplan.bean.TimedTask;
import com.devplatform.equipment.modules.eppatrolplan.model.EpPatrolPlanModel;
import com.devplatform.equipment.modules.eppatrolplan.service.EpPatrolPlanService;
import com.devplatform.equipment.modules.eppatrolplan.service.TimedTaskService;
import com.devplatform.equipment.modules.eppatrolplanpoint.bean.EpPatrolPlanPoint;
import com.devplatform.equipment.modules.eppatrolplanpoint.service.EpPatrolPlanPointService;
import com.devplatform.equipment.modules.eppatrolresult.service.EpPatrolResultService;
import com.devplatform.equipment.modules.listation.bean.LiStation;
import com.devplatform.equipment.modules.listation.service.LiStationService;
import com.devplatform.equipment.modules.sysparams.bean.SysParams;
import com.devplatform.equipment.modules.sysparams.service.SysParamsService;
import com.devplatform.equipment.modules.sysuser.bean.SysUser;
import com.devplatform.equipment.modules.sysuser.service.SysUserService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import lombok.SneakyThrows;
import org.apache.commons.lang.StringUtils;
import org.quartz.CronScheduleBuilder;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SchedulerFactory;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;
import org.quartz.impl.StdSchedulerFactory;
import org.quartz.impl.matchers.GroupMatcher;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * @author Administrator
 */
@Api(tags = {"电子巡查计划表接口"})
@RestController
@RequestMapping("/epPatrolPlan")
public class EpPatrolPlanController extends AbstractController {

    private static SchedulerFactory gSchedulerFactory = new StdSchedulerFactory();
    private static String JOB_GROUP_NAME = "quartz_name";
    private static String TRIGGER_GROUP_NAME = "quartz_group";
    @Autowired
    private EpPatrolPlanService epPatrolPlanService;
    @Autowired
    private EpPatrolResultService epPatrolResultService;
    @Autowired
    private EpPatrolPlanPointService epPatrolPlanPointService;
    @Autowired
    private LiStationService liStationService;
    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private TimedTaskService timedTaskService;
    @Autowired
    private SysParamsService sysParamsService;

    /**
     * @param jobName @Description 移除一个任务(使用默认的任务组名 ， 触发器名 ， 触发器组名)
     */
    public static void removeJob(String jobName) {
        TriggerKey triggerKey = TriggerKey.triggerKey(jobName, TRIGGER_GROUP_NAME);
        JobKey jobKey = JobKey.jobKey(jobName, JOB_GROUP_NAME);
        try {
            Scheduler sched = gSchedulerFactory.getScheduler();
            Trigger trigger = (Trigger) sched.getTrigger(triggerKey);
            if (trigger == null) {
                return;
            }
            // 停止触发器
            sched.pauseTrigger(triggerKey);
            // 移除触发器
            sched.unscheduleJob(triggerKey);
            // 删除任务
            sched.deleteJob(jobKey);
            System.err.println("移除任务:" + jobName);
        } catch (Exception ignored) {
            System.err.println("移除定时任务出错");
        }
    }

    /**
     * 列表页面列表数据获取
     *
     * @param model 承接对象 stationId 站点id name 计划名称 lineName 巡查路线
     * @return
     */
    @ApiOperation(value = "根据条件获取电子巡查计划表分页数据列表", notes = "根据条件获取电子巡查计划表分页数据列表")
    @ApiImplicitParam(name = "model", value = "参数", required = true, dataType = "EpPatrolPlanModel")
    @PostMapping("/list")
    public R list(@RequestBody EpPatrolPlanModel model) {
        if (StringUtil.isEmpty(model.getSort())) {
            model.setSort("create_time");
            model.setOrder("DESC");
        }
        List<EpPatrolPlan> list = epPatrolPlanService.queryPageByList(model);
        return R.ok().put("page", getMyPage(list, model));
    }

    /**
     * 添加
     *
     * @param model 电子巡查计划表对象 stationId 站点id lineId 线路id lineName 巡查线路名 name 计划名称 patrollerId 巡查员id
     *              patrollerName 巡查员名字 startTime 开始日期 endTime 结束日期 circle 巡查周期（单位 ：天 下拉框的值 一天 一周 一个月） remark
     *              备注 pointList 巡查点相关数据集合（格式： pointId 巡查点id pointName 巡查点名字 arrivalTime 到达时间）
     * @return
     */
    @SysLogMethod(operation = "新增", blockName = "电子巡更", detail = "新增电子巡查计划数据")
    @ApiOperation(value = "新增电子巡查计划表数据", notes = "新增电子巡查计划表数据")
    @PostMapping("/save")
    public R save(
            @RequestBody @ApiParam(value = "电子巡查计划表实体对象", required = true) EpPatrolPlanModel model) {
        List<Map<String, String>> pointList = model.getPointList();
        // 判断结束日期是否大于开始日期且都不为空，不是则返回错误信息
        if (!(model.getStartTime() != null
                && model.getEndTime() != null
                && model.getEndTime().getTime() > model.getStartTime().getTime())) {
            return R.error("计划开始或结束日期不符合规范");
        }
        // 判断当前站点有没有相同计划名的数据，有则提示计划名已被使用
        if (epPatrolPlanService.countByStationIdAndName(model.getStationId(), model.getName()) > 0) {
            return R.error("计划名已被使用");
        }
        // 判断当前要保存的计划是否有巡查点 ，有则继续
        if (pointList != null && pointList.size() > 0) {
            // 判断每个巡查点有没有填到达时间，没有提示
            for (Map<String, String> map : pointList) {
                if (StringUtil.isEmpty(map.get("arrivalTime"))) {
                    return R.error("巡查点到达时间必填");
                }
            }
            EpPatrolPlan bean = new EpPatrolPlan();
            BeanUtils.copyProperties(model, bean);
            LiStation station = liStationService.getById(bean.getStationId());
            bean.setLineStationName(station.getLineName() + station.getStationName());
            bean.setCreateUserId(getUserId());
            bean.setCreateTime(new Date());
            bean.setSysSign(getUser().getSysSystem().getCode());
            // 级联添加
            epPatrolPlanService.saveCascade(bean, pointList);
        }
        return R.ok();
    }

    /**
     * 修改
     *
     * @param model 电子巡查计划表对象 TODO 目前不能修改巡查点信息
     * @return
     */
    @SysLogMethod(operation = "修改", blockName = "电子巡更", detail = "修改电子巡查计划数据")
    @ApiOperation(value = "修改电子巡查计划表数据", notes = "修改电子巡查计划表数据")
    @PostMapping("/update")
    public R update(
            @RequestBody @ApiParam(value = "电子巡查计划表实体对象", required = true) EpPatrolPlanModel model) {
        if (StringUtils.isBlank(model.getId())) {
            return R.error("参数有误！");
        }

        List<Map<String, String>> pointList = model.getPointList();
        // 判断结束日期是否大于开始日期且都不为空，不是则返回错误信息
        if (!(model.getStartTime() != null
                && model.getEndTime() != null
                && model.getEndTime().getTime() > model.getStartTime().getTime())) {
            return R.error("计划开始或结束日期不符合规范");
        }
        // 判断当前站点有没有相同计划名的数据，有则提示计划名已被使用
        if (epPatrolPlanService.countByStationIdAndNameAndId(
                model.getStationId(), model.getName(), model.getId())
                > 0) {
            return R.error("计划名已被使用");
        }

        // 判断每个巡查点有没有填到达时间，没有提示
        if (pointList != null && pointList.size() > 0) {
            for (Map<String, String> map : pointList) {
                if (StringUtil.isEmpty(map.get("arrivalTime"))) {
                    return R.error("巡查点到达时间必填");
                }
            }
        }
        EpPatrolPlan bean = new EpPatrolPlan();
        BeanUtils.copyProperties(model, bean);
        LiStation station = liStationService.getById(bean.getStationId());
        bean.setLineStationName(station.getLineName() + station.getStationName());
        bean.setUpdateUserId(getUserId());
        bean.setUpdateTime(new Date());
        bean.setSysSign(getUser().getSysSystem().getCode());
        // 级联修改
        epPatrolPlanService.updateCascade(bean, pointList);

        return R.ok();
    }

    /**
     * 测试巡查计划是否可以修改
     *
     * @return
     */
    @GetMapping("/checkToUpdate")
    public R checkToUpdate(String id, String stationId) {
        if (StringUtils.isNotBlank(id)) {
            /// 判断此巡查计划有没有巡查结果，有则不能修改 ，且给提示信息
            if (epPatrolResultService.countByPlanId(id, stationId) <= 0) {
                return R.ok();
            }
        }
        return R.error("此巡查计划已被执行过，不能修改");
    }

    /**
     * 根据ID获取电子巡查计划表详情对象
     *
     * @param id 对象主键
     * @return
     */
    @ApiOperation(value = "根据ID获取电子巡查计划表详情对象", notes = "根据ID获取电子巡查计划表详情对象")
    @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
    @GetMapping("/getId")
    public R getId(String id, String stationId) {
        EpPatrolPlan epPatrolPlan =
                epPatrolPlanService.getOne(
                        new LambdaQueryWrapper<EpPatrolPlan>()
                                .eq(EpPatrolPlan::getId, id)
                                .eq(StringUtil.checkNotNull(stationId), EpPatrolPlan::getStationId, stationId));
        // 获取相关巡查点信息
        List<EpPatrolPlanPoint> list = epPatrolPlanPointService.getByPlanId(id, stationId);
        Map map = new LinkedHashMap<String, Object>();
        map.put("id", epPatrolPlan.getId());
        map.put("stationId", epPatrolPlan.getStationId());
        map.put("lineId", epPatrolPlan.getLineId());
        map.put("lineName", epPatrolPlan.getLineName());
        map.put("name", epPatrolPlan.getName());
        map.put("patrollerId", epPatrolPlan.getPatrollerId());
        map.put("patrollerName", epPatrolPlan.getPatrollerName());
        map.put("startTime", epPatrolPlan.getStartTime());
        map.put("endTime", epPatrolPlan.getEndTime());
        map.put("circle", epPatrolPlan.getCircle());
        map.put("remark", epPatrolPlan.getRemark());
        ArrayList<Map<String, String>> pointList = new ArrayList<>();
        for (EpPatrolPlanPoint epPatrolPlanPoint : list) {
            Map<String, String> lmap = new LinkedHashMap<>();
            lmap.put("id", epPatrolPlanPoint.getId());
            lmap.put("pointId", epPatrolPlanPoint.getPointId());
            lmap.put("arrivalTime", epPatrolPlanPoint.getArrivalTime());
            lmap.put("pointName", epPatrolPlanPoint.getPointName());
            pointList.add(lmap);
        }
        map.put("pointList", pointList);
        return R.ok().put("bean", map);
    }

    /**
     * 根据ID删除电子巡查计划表对象
     *
     * @param id 电子巡查计划表对象主键数组
     * @return
     */
    @ApiOperation(value = "根据ID删除电子巡查计划表数据", notes = "根据ID删除电子巡查计划表数据")
    @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
    @GetMapping("/delete")
    public R delete(String id, String stationId) {
        if (StringUtil.checkNotNull(id)) {
            /// 判断此巡查计划有没有巡查结果，有则不能删除 ，且给提示信息
            if (epPatrolResultService.countByPlanId(id, stationId) <= 0) {
                // 级联删除
                epPatrolPlanService.deleteById(id, stationId, getUser().getSysSystem().getCode());
            } else {
                return R.error("此巡查计划已被执行过，不能删除");
            }
        }
        return R.ok();
    }

    @ApiOperation(value = "获取巡查点波动时间", notes = "ID获取巡查点波动时间")
    @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
    @GetMapping("/getFluctuationTime")
    public R getFluctuationTime(String stationId) {
        SysParams bean = sysParamsService.getOne(new LambdaQueryWrapper<SysParams>().eq(SysParams::getName, "patrolTime").eq(SysParams::getStationId, stationId));
        return R.ok().put("time", bean.getValue());
    }

    @ApiOperation(value = "发送邮件提醒巡查员", notes = "发送邮件提醒巡查员")
    @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
    @PostMapping("/sendMail")
    public R sendMail(String id, String stationId, String sendTime, String isReboot) throws ParseException, SchedulerException {
        if (StringUtil.checkNotNull(id)) {
            Scheduler sched = gSchedulerFactory.getScheduler();

            //查询巡查计划
            LambdaQueryWrapper<EpPatrolPlan> eq = new LambdaQueryWrapper<EpPatrolPlan>()
                    .eq(EpPatrolPlan::getId, id)
                    .eq(EpPatrolPlan::getStationId, stationId);
            EpPatrolPlan plan = epPatrolPlanService.getOne(eq);
            if (plan == null || StringUtils.isBlank(plan.getPatrollerId())) {
                return R.error();
            }

            // 错过开始时间（结束时间在现在之前）
            boolean after = plan.getEndTime().before(new Date());
            LambdaQueryWrapper<TimedTask> queryWrapper = new LambdaQueryWrapper<TimedTask>()
                    .eq(TimedTask::getPlanId, id)
                    .eq(TimedTask::getDeleted, 0)
                    .eq(StringUtil.checkNotNull(stationId), TimedTask::getStationId, stationId);
            if (after) {
                // 重启就错过了时机，直接跳过定时任务，设置执行状态为失败
                if (StringUtils.isNotBlank(isReboot)) {
                    TimedTask tt = new TimedTask();
                    tt.setStatus(1);
                    tt.setDeleted(1);
                    timedTaskService.update(tt, queryWrapper);
                }
                return R.error("错过开始时间！");
            }

            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

            // 删除对应的定时任务
            removeJob(id);

            //设置定时任务
            R error = setTimeTask(id, stationId, sendTime, sched, plan, queryWrapper, sdf);
            if (error != null) {
                return error;
            }

            // 持久化到数据库
            // 重启不需要持久化
            if (StringUtils.isBlank(isReboot)) {
                SysUserEntity user = getUser();
                TimedTask timedTask = new TimedTask();
                timedTask.setPlanId(id);
                timedTask.setSendTime(sdf.parse(sendTime));
                timedTask.setCreateTime(new Date());
                timedTask.setCreateUserId(user.getUserId());
                timedTask.setSysSign(user.getSysSystem().getCode());
                timedTask.setStationId(user.getSysSystem().getCode());

                TimedTask one = timedTaskService.getOne(queryWrapper);
                if (one == null || StringUtils.isBlank(one.getId())) {
                    timedTaskService.save(timedTask);
                } else {
                    timedTaskService.update(timedTask, queryWrapper);
                }
            }
        }
        return R.ok();
    }

    /**
     * 设置定时任务
     *
     * @param id
     * @param stationId
     * @param sendTime
     * @param sched
     * @param plan
     * @param queryWrapper
     * @param sdf
     * @return
     * @throws ParseException
     */
    private R setTimeTask(String id, String stationId, String sendTime, Scheduler sched, EpPatrolPlan plan, LambdaQueryWrapper<TimedTask> queryWrapper, SimpleDateFormat sdf) throws ParseException {
        //查询巡查人员信息
        LambdaQueryWrapper<SysUser> eq1 = new LambdaQueryWrapper<SysUser>()
                .eq(SysUser::getUserId, plan.getPatrollerId())
                .eq(StringUtil.checkNotNull(stationId), SysUser::getStationId, stationId);
        SysUser sysUser = sysUserService.getOne(eq1);
        JobDataMap resJobDataMap = new JobDataMap();
        resJobDataMap.put("mail", sysUser.getEmail());
        resJobDataMap.put("plan", JSON.toJSONString(plan));

        JobDetail job = JobBuilder.newJob(SendMessage.class).withIdentity(id, JOB_GROUP_NAME).usingJobData(resJobDataMap).build();

        String cron = createCron(sendTime);
        Trigger trigger = TriggerBuilder.newTrigger()
                .withIdentity(id, TRIGGER_GROUP_NAME)
                .startAt(plan.getStartTime())
                .withSchedule(CronScheduleBuilder.cronSchedule(cron).withMisfireHandlingInstructionDoNothing()).endAt(plan.getEndTime()).build();
        // 交给scheduler去调度
        try {
            sched.scheduleJob(job, trigger);
            // 启动
            if (!sched.isShutdown()) {
                sched.start();
                TimedTask ttt = new TimedTask();
                ttt.setStatus(1);
                timedTaskService.update(ttt, queryWrapper);
                System.err.println("设置定时任务：" + id + "\n\t下次运行时间：" + sdf.format(trigger.getNextFireTime()));
            }
        } catch (SchedulerException e) {
            e.printStackTrace();
            return R.error("时间设置有误！");
        }
        return null;
    }

    /**
     * 生成cron表达
     *
     * @param sendTime 发送时间
     * @return cron
     * @throws ParseException
     */
    public String createCron(String sendTime) throws ParseException {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        Calendar ca1 = Calendar.getInstance();
        ca1.setTime(sdf.parse(sendTime));
        int hour = ca1.get(Calendar.HOUR_OF_DAY);
        int minute = ca1.get(Calendar.MINUTE);
        int second = ca1.get(Calendar.SECOND);
        StringBuilder sb = new StringBuilder()
                .append(second).append(" ")
                .append(minute).append(" ")
                .append(hour).append(" * * ? ");
        return sb.toString();
    }

    /**
     * 查看定时任务执行状态
     *
     * @throws SchedulerException
     */
    @GetMapping("getAllJob")
    @SneakyThrows
    void getAllJob() throws SchedulerException {
        Scheduler sched = gSchedulerFactory.getScheduler();

        for (String groupName : sched.getJobGroupNames()) {

            for (JobKey jobKey : sched.getJobKeys(GroupMatcher.jobGroupEquals(groupName))) {

                String jobName = jobKey.getName();
                String jobGroup = jobKey.getGroup();

                // get job's trigger
                List<Trigger> triggers = (List<Trigger>) sched.getTriggersOfJob(jobKey);
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                Date nextFireTime = triggers.get(0).getNextFireTime();

                System.err.println(
                        "[jobName] : "
                                + jobName
                                + " [groupName] : "
                                + jobGroup
                                + " - [下次执行时间] : "
                                + sdf.format(nextFireTime));
            }
        }
    }
}
