package com.devplatform.equipment.modules.syspreset.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.devplatform.common.base.annotation.SysLog;
import com.devplatform.common.base.validator.ValidatorUtils;
import com.devplatform.common.util.R;
import com.devplatform.common.util.StringUtil;
import com.devplatform.equipment.common.annotation.SysLogMethod;
import com.devplatform.equipment.common.utils.AbstractController;
import com.devplatform.equipment.common.utils.Constants;
import com.devplatform.equipment.modules.liresource.bean.LiResource;
import com.devplatform.equipment.modules.liresource.service.LiResourceService;
import com.devplatform.equipment.modules.syspreset.bean.SysPreset;
import com.devplatform.equipment.modules.syspreset.model.SysPresetModel;
import com.devplatform.equipment.modules.syspreset.service.SysPresetService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
/**
 * @author Administrator
 */
@Api(tags = {"预置点管理接口"})
@RestController
@RequestMapping("/sysPreset")
public class SysPresetController extends AbstractController {

  @Autowired private SysPresetService sysPresetService;

  @Autowired private LiResourceService liResourceService;

  /**
   * 列表页面列表数据获取
   *
   * @param model 承接对象 page 页码 rows 每页条数 resourceId 资源点id
   * @return
   */
  @ApiOperation(value = "根据条件获取预置点管理分页数据列表", notes = "根据条件获取预置点管理分页数据列表")
  @ApiImplicitParam(name = "model", value = "参数", required = true, dataType = "SysPresetModel")
  @PostMapping("/list")
  public R list(@RequestBody SysPresetModel model) {
    if (StringUtils.isBlank(model.getSort())) {
      model.setSort("order_num");
      model.setOrder("ASC");
    }
    List<SysPreset> list = sysPresetService.queryPageByList(model);
    LiResource liResource = liResourceService.getBeanById(model.getResourceId());
    String resourceName = liResource.getName();
    String resourceCode = liResource.getCode();
    ArrayList<Map<String, String>> mapList = new ArrayList<>();
    list.forEach(
        (sysPreset) -> {
          Map map = JSONObject.parseObject(JSONObject.toJSONString(sysPreset), Map.class);
          map.remove("resourceId");
          map.put("resourceName", resourceName);
          map.put("resourceCode", resourceCode);
          mapList.add(map);
        });
    return R.ok().put("page", getMyPage(mapList, model));
  }

  /**
   * 添加
   *
   * @param bean 预置点管理对象 resourceId 资源点id name 预置点名称 orderNum 排序
   * @return
   */
  @SysLog("添加预置点管理")
  @SysLogMethod(operation = "新增", blockName = "预置点管理", detail = "新增预置点管理数据")
  @ApiOperation(value = "新增预置点管理数据", notes = "新增预置点管理数据")
  @PostMapping("/save")
  public R save(@RequestBody @ApiParam(value = "预置点管理实体对象", required = true) SysPreset bean) {
    ValidatorUtils.validateEntity(bean);
    // 判断编号是否是1-99，并且没有被使用
    if (bean.getOrderNum() < 1 || bean.getOrderNum() > Constants.INT_99) {
      return R.error("预置点编号必须为1-99的数");
    }
    if (sysPresetService.countByResourceId(
            null, bean.getResourceId(), bean.getOrderNum(), bean.getStationId())
        > 0) {
      return R.error("预置点编号已被使用");
    }
    bean.setSysSign(getUser().getSysSystem().getCode());
    sysPresetService.save(bean);
    return R.ok();
  }

  /**
   * 修改
   *
   * @param bean 预置点管理对象
   * @return
   */
  @SysLog("修改预置点管理")
  @SysLogMethod(operation = "修改", blockName = "预置点管理", detail = "修改预置点管理数据")
  @ApiOperation(value = "修改预置点管理数据", notes = "修改预置点管理数据")
  @PostMapping("/update")
  public R update(@RequestBody @ApiParam(value = "预置点管理实体对象", required = true) SysPreset bean) {
    ValidatorUtils.validateEntity(bean);
    bean.setSysSign(getUser().getSysSystem().getCode());
    // 判断编号是否是1-99，并且没有被使用
    if (bean.getOrderNum() < 1 || bean.getOrderNum() > Constants.INT_99) {
      return R.error("预置点编号必须为1-99的数");
    }
    if (sysPresetService.countByResourceId(
            bean.getId(), bean.getResourceId(), bean.getOrderNum(), bean.getStationId())
        > 0) {
      return R.error("预置点编号已被使用");
    }
    sysPresetService.update(
        bean,
        new LambdaQueryWrapper<SysPreset>()
            .eq(SysPreset::getId, bean.getId())
            .eq(
                StringUtil.checkNotNull(bean.getStationId()),
                SysPreset::getStationId,
                bean.getStationId()));
    return R.ok();
  }

  /**
   * 根据ID获取预置点管理对象
   *
   * @param id 对象主键
   * @return
   */
  @SysLog("删除预置点管理")
  @SysLogMethod(operation = "删除", blockName = "预置点管理", detail = "删除预置点管理数据")
  @ApiOperation(value = "根据ID删除预置点管理对象", notes = "根据ID删除预置点管理对象")
  @ApiImplicitParam(name = "id", value = "主键", required = true, dataType = "String")
  @GetMapping("/delete")
  public R delete(String id, String stationId) {
    // 逻辑删除
    SysPreset bean = new SysPreset();
    bean.setId(id);
    bean.setDeleted(1);
    bean.setSysSign(getUser().getSysSystem().getCode());
    sysPresetService.update(
        bean,
        new QueryWrapper<SysPreset>()
            .eq("id", id)
            .eq(StringUtil.checkNotNull(stationId), "station_id", stationId));
    return R.ok();
  }

  @ApiOperation(value = "根据资源id获取预置点管理对象", notes = "根据资源id获取预置点管理对象")
  @ApiImplicitParam(name = "resourceId", value = "资源id", required = true, dataType = "String")
  @GetMapping("/getByResourceId")
  public R getByResourceId(String resourceId, String stationId) {
    List<SysPreset> list =
        sysPresetService.list(
            new QueryWrapper<SysPreset>()
                .eq("resource_id", resourceId)
                .eq("deleted", 0)
                .eq(StringUtil.checkNotNull(stationId), "station_id", stationId)
                .orderByAsc("order_num"));
    return R.ok().put("list", list);
  }

  /**
   * 根据资源点id页面列表数据获取
   *
   * @param model 承接对象 resourceId 资源点id
   * @return
   */
  @ApiOperation(value = "根据条件获取预置点管理分页数据列表", notes = "根据条件获取预置点管理分页数据列表")
  @ApiImplicitParam(name = "model", value = "参数", required = true, dataType = "SysPresetModel")
  @PostMapping("/queryByResourceId")
  public R queryByResourceId(@RequestBody SysPresetModel model) {
    if (StringUtils.isBlank(model.getSort())) {
      model.setSort("order_num");
      model.setOrder("ASC");
    }
    List<SysPreset> list = sysPresetService.queryPageByList(model);
    return R.ok().put("page", list);
  }
}
