package com.devplatform.admin.common.aspect;

import com.alibaba.fastjson.JSON;
import com.devplatform.admin.common.annotation.SysLogMethod;
import com.devplatform.admin.common.utils.AbstractController;
import com.devplatform.admin.common.utils.Constants;
import com.devplatform.admin.common.utils.HttpContextUtils;
import com.devplatform.admin.common.utils.IpUtils;
import com.devplatform.admin.modules.sys.bean.SysLogEntity;
import com.devplatform.admin.modules.sys.bean.SysUserEntity;
import com.devplatform.admin.modules.sys.service.SysLogService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.shiro.SecurityUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.Date;

/**
 * 系统日志，切面处理类
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2017年3月8日 上午11:07:35
 */
@Aspect
@Component
public class SysLogAspect extends AbstractController {

  private static Log log = LogFactory.getLog(SysLogAspect.class);

  @Autowired private SysLogService sysLogService;

  @Pointcut(
      "@annotation(com.devplatform.admin.common.annotation.SysLogMethod) && execution(* com.devplatform.admin..*.*(..))")
  public void logPointCut() {
    log.info("");
  }

  @Around("logPointCut(),logMethodPointCut()")
  public Object around(ProceedingJoinPoint point) throws Throwable {
    long beginTime = System.currentTimeMillis();
    // 执行方法
    Object result = point.proceed();
    // 执行时长(毫秒)
    long time = System.currentTimeMillis() - beginTime;

    // 保存日志
    saveSysLog(point, time);

    return result;
  }

  private void saveSysLog(ProceedingJoinPoint joinPoint, long time) {
    MethodSignature signature = (MethodSignature) joinPoint.getSignature();
    Method method = signature.getMethod();

    SysLogEntity sysLog = new SysLogEntity();
    SysLogMethod sysLogMethod = method.getAnnotation(SysLogMethod.class);
    if (sysLogMethod != null) {
      // 注解上的描述
      sysLog.setOperation(sysLogMethod.operation());
      sysLog.setBlockName(sysLogMethod.blockName());
      sysLog.setDetail(sysLogMethod.detail());
    }

    // 请求的方法名
    String className = joinPoint.getTarget().getClass().getName();
    String methodName = signature.getName();
    sysLog.setMethod(className + "." + methodName + "()");

    // 请求的参数
    Object[] args = joinPoint.getArgs();
    try {
      String params = JSON.toJSONString(args);
      sysLog.setParams(params);
    } catch (Exception e) {

    }
    // 获取request
    HttpServletRequest request = HttpContextUtils.getHttpServletRequest();
    // 设置IP地址
    sysLog.setIp(IpUtils.getIpAddr(request));
    // 用户名
    SysUserEntity user = ((SysUserEntity) SecurityUtils.getSubject().getPrincipal());
    String stationId = request.getHeader("stationId");
    if (Constants.STRING_1.equals(user.getSysSystem().getType()) && !Constants.CHECK.equals(sysLogMethod.operation())) {
      sysLog.setUsername(user.getName());
      sysLog.setStationId(user.getSysSystem().getCode());
    } else if (Constants.STRING_2.equals(user.getSysSystem().getType())) {
      sysLog.setUsername("线路管理员");
      sysLog.setStationId(stationId);
    } else if (Constants.STRING_3.equals(user.getSysSystem().getType())) {
      sysLog.setUsername("路网管理员");
      sysLog.setStationId(stationId);
    } else if (Constants.STRING_1.equals(user.getSysSystem().getType()) && Constants.CHECK.equals(sysLogMethod.operation())) {
      // 站点自己访问时不记录查看日志
      return;
    }
    sysLog.setUserId(user.getUserId());
    sysLog.setOrgId(user.getOrgId());
    sysLog.setTime(time);
    sysLog.setCreateDate(new Date());
    sysLog.setSysSign(getUser().getSysSystem().getCode());
    sysLog.setCreateBy(getUserId());

    // 保存系统日志
    sysLogService.save(sysLog);
  }
}
