package com.devplatform.admin.common.utils;

import com.google.gson.Gson;
import java.util.concurrent.TimeUnit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.SetOperations;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.stereotype.Component;

/**
 * Redis工具类
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2017-07-17 21:12
 */
@Component
public class RedisUtils {
  /** 默认过期时长，单位：秒 */
  public static final long DEFAULT_EXPIRE = 60L * 60L * 24L;
  /** 不设置过期时长 */
  public static final long NOT_EXPIRE = -1;
  private static final Gson GSON = new Gson();
  @Autowired private RedisTemplate<String, Object> redisTemplate;
  @Autowired private ValueOperations<String, String> valueOperations;
  @Autowired private HashOperations<String, String, Object> hashOperations;
  @Autowired private ListOperations<String, Object> listOperations;
  @Autowired private SetOperations<String, Object> setOperations;
  @Autowired private ZSetOperations<String, Object> zSetOperations;

  public void set(String key, Object value, long expire) {
    valueOperations.set(key, toJson(value));
    if (expire != NOT_EXPIRE) {
      redisTemplate.expire(key, expire, TimeUnit.SECONDS);
    }
  }

  public void set(String key, Object value) {
    set(key, value, DEFAULT_EXPIRE);
  }

  public <T> T get(String key, Class<T> clazz, long expire) {
    String value = valueOperations.get(key);
    if (expire != NOT_EXPIRE) {
      redisTemplate.expire(key, expire, TimeUnit.SECONDS);
    }
    return value == null ? null : fromJson(value, clazz);
  }

  public <T> T get(String key, Class<T> clazz) {
    return get(key, clazz, NOT_EXPIRE);
  }

  public String get(String key, long expire) {
    String value = valueOperations.get(key);
    if (expire != NOT_EXPIRE) {
      redisTemplate.expire(key, expire, TimeUnit.SECONDS);
    }
    return value;
  }

  public String get(String key) {
    return get(key, NOT_EXPIRE);
  }

  public void delete(String key) {
    redisTemplate.delete(key);
  }

  /** Object转成JSON数据 */
  private String toJson(Object object) {
    if (object instanceof Integer
        || object instanceof Long
        || object instanceof Float
        || object instanceof Double
        || object instanceof Boolean
        || object instanceof String) {
      return String.valueOf(object);
    }
    return GSON.toJson(object);
  }

  /** JSON数据，转成Object */
  private <T> T fromJson(String json, Class<T> clazz) {
    return GSON.fromJson(json, clazz);
  }
}
