package com.devplatform.admin.common.utils;

import com.devplatform.common.base.exception.RRException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.xssf.usermodel.*;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * @author Muceball-laptop
 * @version 1.0
 * @date 2021/2/23 11:50
 */
@Slf4j
public class XlsParseUtil {
    private final String xlsFileName;
    private final XSSFWorkbook workbook;
    private final XSSFSheet sheet;
    private final XSSFDrawing drawing;
    // private final XSSFCellStyle whiteStyle;
    // private final XSSFCellStyle yellowStyle;
    private String path;

    /**
     * 初始化Excel
     *
     * @param fileName 导出文件名
     * @param path     导出文件路径
     * @param ins      excel文件流
     */
    public XlsParseUtil(String fileName, String path, InputStream ins) {
        //文件名
        this.xlsFileName = fileName;
        //路径
        this.path = path;
        try {
            this.workbook = new XSSFWorkbook(ins);
            this.sheet = workbook.getSheetAt(0);
        } catch (IOException e) {
            log.error("解析文件失败");
            throw new RRException(e.getMessage());
        } finally {
            if (ins != null) {
                try {
                    ins.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        this.drawing = sheet.createDrawingPatriarch();
    }

    /**
     * 获取最大行数
     *
     * @param
     * @return java.lang.Integer
     * @author dhl
     * @date 2021/2/24 0024 9:49
     */
    public Integer getLastRowNum() {
        return this.sheet.getLastRowNum();
    }

    /**
     * 获取当前行的最大列数
     *
     * @param rowNum
     * @return java.lang.Integer
     * @author dhl
     * @date 2021/3/1 0001 10:43
     */
    public Integer getLastCellNum(int rowNum) {
        return Integer.valueOf(this.sheet.getRow(rowNum).getLastCellNum());
    }

    /**
     * 获取单元格数据
     *
     * @param rowNum 行号（从0开始）
     * @param colNum 列号（从0开始）
     * @param c      返回值类型
     * @return 单元格数据
     */
    @SuppressWarnings("unchecked")
    public <T> T getCellValue(int rowNum, int colNum, Class<T> c) {
        XSSFRow row = sheet.getRow(rowNum);
        if (row == null) {
            return null;
        }
        XSSFCell cell = row.getCell(colNum);
        if (cell == null) {
            return null;
        }
        if (c == null || String.class.isAssignableFrom(c)) {
            return (T) cell.getStringCellValue();
        } else if (Date.class.isAssignableFrom(c)) {
            return (T) cell.getDateCellValue();
        } else if (boolean.class.isAssignableFrom(c) || Boolean.class.isAssignableFrom(c)) {
            return (T) Boolean.valueOf(cell.getBooleanCellValue());
        } else if (Integer.class.isAssignableFrom(c) || int.class.isAssignableFrom(c)) {
            double numericCellValue = cell.getNumericCellValue();
            return (T) Integer.valueOf(new Double(numericCellValue).intValue());
        } else if (Double.class.isAssignableFrom(c) || double.class.isAssignableFrom(c)) {
            return (T) Double.valueOf(cell.getNumericCellValue());
        } else {
            return null;
        }
    }

    /**
     * 获取单元格数据
     *
     * @param rowNum 行号（从0开始）
     * @param colNum 列号（从0开始）
     * @return 单元格数据
     */
    public String getCellValue(int rowNum, int colNum) {
        XSSFRow row = sheet.getRow(rowNum);
        if (row == null) {
            return null;
        }
        XSSFCell cell = row.getCell(colNum);
        if (cell == null) {
            return null;
        }

        String cellValue = "";
        // 以下是判断数据的类型
        CellType cellType = cell.getCellTypeEnum();
        if (cellType == CellType.NUMERIC) { // 数字
            if (DateUtil.isCellDateFormatted(cell)) {
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                Date data = cell.getDateCellValue();
                cellValue = sdf.format(data);
            } else {
                // cellValue = String.valueOf(cell.getNumericCellValue());
                cell.setCellType(CellType.STRING);
                cellValue = cell.getStringCellValue();
            }
        } else if (cellType == CellType.STRING) {
            // 字符串
            cellValue = cell.getStringCellValue();
        } else if (cellType == CellType.BOOLEAN) {
            // Boolean
            cellValue = String.valueOf(cell.getBooleanCellValue());
        } else if (cellType == CellType.FORMULA) {
            // 公式
            cellValue = cell.getCellFormula();
        } else if (cellType == CellType.BLANK) {
            // 空值
            cellValue = "";
        } else if (cellType == CellType.ERROR) {
            // 故障
            cellValue = null;
        } else {
            cellValue = null;
        }
        return cellValue;
    }

    /**
     * 修改文件中的批注信息
     *
     * @param rowNum     行号（从0开始）
     * @param colNum     列号（从0开始）
     * @param commentStr 批注信息
     */
    public void updateComment(int rowNum, int colNum, String commentStr) {
        XSSFRow row = sheet.getRow(rowNum);
        XSSFCell cell;
        if (row == null) {
            row = sheet.createRow(rowNum);
        }
        cell = row.getCell(colNum);
        if (cell == null) {
            cell = row.createCell(colNum);
            cell.setCellValue("");
            cell.setCellType(CellType.STRING);
        }
        XSSFCellStyle cellStyle = cell.getCellStyle();
        if (StringUtils.isBlank(commentStr)) {
            cell.removeCellComment();
            // setStyle(cell, whiteStyle, IndexedColors.WHITE.getIndex());
            if (cellStyle == null) {
                XSSFCellStyle style = workbook.createCellStyle();
                style.setFillPattern(FillPatternType.NO_FILL);
                cell.setCellStyle(style);
            } else {
                CellStyle newStyle = workbook.createCellStyle();
                // 克隆出一个 style
                newStyle.cloneStyleFrom(cellStyle);
                newStyle.setFillPattern(FillPatternType.NO_FILL);
                cell.setCellStyle(newStyle);
            }
        } else {
            Comment comment = cell.getCellComment();
            if (comment == null) {
                //批注为3行4列大小
                comment = drawing.createCellComment(new XSSFClientAnchor(0, 0, 0, 0, colNum, rowNum, colNum + 3, rowNum + 2));
            }
            // 输入批注信息
            comment.setString(new XSSFRichTextString(commentStr));
            // 添加作者
            comment.setAuthor("Suntray");
            // 将批注添加到单元格对象中
            cell.setCellComment(comment);
            if (cellStyle == null) {
                XSSFCellStyle style = workbook.createCellStyle();
                style.setFillForegroundColor(IndexedColors.GREEN.getIndex());
                style.setFillPattern(FillPatternType.NO_FILL);
                cell.setCellStyle(style);
            } else {
                // CellStyle newStyle = cell.getRow().getSheet().getWorkbook().createCellStyle();
                CellStyle newStyle = workbook.createCellStyle();
                // 克隆出一个 style
                newStyle.cloneStyleFrom(cellStyle);
                newStyle.setFillForegroundColor(IndexedColors.YELLOW.getIndex());
                newStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
                cell.setCellStyle(newStyle);
            }
        }
    }

    /**
     * 导出Excel文件
     */
    @SuppressWarnings("all")
    public void exportXlS() throws Exception {
        if (StringUtils.isNotBlank(path) && !path.endsWith("/")) {
            path = path + "/";
        }
        File file = new File(path);
        if (!file.exists() && !file.isDirectory()) {
            log.info("创建文件夹：" + path);
            file.mkdirs();
        }
        try (FileOutputStream out = new FileOutputStream(path + xlsFileName)) {
            workbook.write(out);
        } finally {
            workbook.close();
        }
    }
}
