package com.devplatform.admin.modules.sys.controller;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.devplatform.admin.common.annotation.SysLogMethod;
import com.devplatform.admin.common.utils.*;
import com.devplatform.admin.modules.sys.bean.SysLogEntity;
import com.devplatform.admin.modules.sys.bean.SysSystem;
import com.devplatform.admin.modules.sys.bean.SysUserEntity;
import com.devplatform.admin.modules.sys.bean.SysUserRoleEntity;
import com.devplatform.admin.modules.sys.form.SysLoginForm;
import com.devplatform.admin.modules.sys.service.*;
import com.devplatform.common.base.exception.RRException;
import com.devplatform.common.util.Constant;
import com.devplatform.common.util.R;
import com.devplatform.common.util.StringUtil;
import com.devplatform.common.util.license.LicenseAuth;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import lombok.SneakyThrows;
import org.apache.commons.io.IOUtils;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import javax.imageio.ImageIO;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.*;

/**
 * 登录相关
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2016年11月10日 下午1:15:31
 */
@Api(tags = {"登陆接口"})
@RestController
public class SysLoginController extends AbstractController {

    @Value("${environment}")
    private String environment;

    @Autowired
    private SysUserService sysUserService;
    @Autowired
    private SysUserRoleService sysUserRoleService;
    @Autowired
    private SysUserTokenService sysUserTokenService;
    @Autowired
    private SysCaptchaService sysCaptchaService;
    @Autowired
    private ShiroService shiroService;
    @Autowired
    private SysLogService sysLogService;
    @Autowired
    private SysSystemService sysSystemService;

    /**
     * 验证码
     */
    @ApiOperation(value = "获取验证码", notes = "获取验证码")
    @ApiImplicitParam(
            name = "uuid",
            value = "uuid",
            required = true,
            paramType = "query",
            dataType = "String")
    @GetMapping("captcha.jpg")
    public void captcha(HttpServletResponse response, String uuid)
            throws ServletException, IOException {
        response.setHeader("Cache-Control", "no-store, no-cache");
        response.setContentType("image/jpeg");

        // 获取图片验证码
        BufferedImage image = sysCaptchaService.getCaptcha(uuid);

        ServletOutputStream out = response.getOutputStream();
        ImageIO.write(image, "jpg", out);
        IOUtils.closeQuietly(out);
    }

    /**
     * 登录
     */
    @SneakyThrows
    @ApiOperation(value = "用户登录", notes = "用户登录")
    @ApiImplicitParam(name = "form", value = "用户实体类", required = true, paramType = "query", dataType = "SysLoginForm")
    @PostMapping("/sys/login")
    public Map<String, Object> login(@RequestBody SysLoginForm form) {
//        String str = LicenseAuth.authLicense(environment);
//        if(!str.equals("YES")) {
//            return R.error(str);
//        }
        boolean captcha = sysCaptchaService.validate(form.getUuid(), form.getCaptcha());
        if(!captcha){
            return R.error("验证码不正确");
        }

        Long startTime = System.currentTimeMillis();
        SysSystem sysSystem = sysSystemService.getById(Constants.CODE);

        String u = Aes.decrypt(form.getUsername());
        String p = Aes.decrypt(form.getPassword());
        // 用户信息
        SysUserEntity user = sysUserService.queryByUserName(u);
        // 账号不存在、密码错误
        if (user == null || !user.getPassword().equals(new Sha256Hash(p, user.getSalt()).toHex())) {
            return R.error("账号或密码不正确");
        }

        // 账号锁定
        if (user.getStatus() == 0) {
            return R.error("账号已被禁用,请联系管理员");
        }

        //		生成token，并保存到数据库
        R r = sysUserTokenService.createToken(user.getUserId());

        Jwts.builder()
                .setSubject(form.getUsername())
                .claim("roles", "member")
                .setIssuedAt(new Date())
                .signWith(SignatureAlgorithm.HS256, "secretkey")
                .compact();

        List<SysUserRoleEntity> userRoleList = sysUserRoleService.list(new QueryWrapper<SysUserRoleEntity>().eq("user_id", user.getUserId()));

        List<String> roleIdList = new ArrayList<String>();
        for (SysUserRoleEntity userRole : userRoleList) {
            roleIdList.add(userRole.getRoleId());
        }
        user.setRoleIdList(roleIdList);
        String jwtToken = null;
        user.setSysSystem(sysSystem);
        try {
            jwtToken =
                    JwtUtil.createJwt("longingJWT", JwtUtil.generalSubject(user), 1000 * 60 * 60 * 24 * 7);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RRException("JWT生成异常");
        }
        // 用户权限列表
        Set<String> permissions = shiroService.getUserPermissions(user.getUserId());

        if (Constant.SUPER_ADMIN.equals(user.getUserId())) {
            // 超级管理员不用判断
        } else if (userRoleList.isEmpty() || permissions.isEmpty()) {
            return R.error("用户登录无权限，请联系管理员！");
        }
        // 记录登录日志
        saveLog("登录", "系统登录", "登录系统成功", "com.devplatform.admin.modules.sys.controller.SysLoginController.login()", form, user, startTime);
        return R.ok()
                .put("token", jwtToken)
                .put("permissions", permissions)
                .put("userId", user.getUserId())
                .put("userName", user.getName())
                .put("sysSystem", JSON.toJSONString(sysSystem));
    }

    private void saveLog(
            String operation,
            String blockName,
            String detail,
            String method,
            Object param,
            SysUserEntity user,
            Long startTime) {
        // 记录登录日志
        SysLogEntity sysLog = new SysLogEntity();
        sysLog.setOperation(operation);
        sysLog.setBlockName(blockName);
        sysLog.setDetail(detail);
        sysLog.setMethod(method);
        try {
            String params = JSON.toJSONString(param);
            sysLog.setParams(params);
        } catch (Exception e) {

        }
        // 获取request
        HttpServletRequest request = HttpContextUtils.getHttpServletRequest();
        // 设置IP地址
        sysLog.setIp(IpUtils.getIpAddr(request));
        sysLog.setUserId(user.getUserId());
        sysLog.setUsername(user.getName());
        sysLog.setOrgId(user.getOrgId());
        Long endTime = System.currentTimeMillis();
        sysLog.setTime(endTime - startTime);
        sysLog.setCreateDate(new Date());
        sysLog.setCreateBy(user.getUserId());
        // 保存系统日志
        sysLogService.save(sysLog);
    }

    /**
     * SSO登录
     */
    @ApiOperation(value = "oauth用户登录", notes = "oauth用户登录")
    @ApiImplicitParam(
            name = "form",
            value = "用户实体类",
            required = true,
            paramType = "query",
            dataType = "SysLoginForm")
    @PostMapping("/sys/sso/{userId}")
    public Map<String, Object> sso(@PathVariable("form") SysLoginForm form) {
        Long startTime = System.currentTimeMillis();
        boolean captcha = sysCaptchaService.validate(form.getUuid(), form.getCaptcha());
        if (!captcha) {
            return R.error("验证码不正确");
        }
        String u = Aes.decrypt(form.getUsername());
        String p = Aes.decrypt(form.getPassword());
        // 用户信息
        SysUserEntity user = sysUserService.queryByUserName(u);
        // 账号不存在、密码错误
        if (user == null || !user.getPassword().equals(new Sha256Hash(p, user.getSalt()).toHex())) {
            return R.error("账号或密码不正确");
        }

        // 账号锁定
        if (user.getStatus() == 0) {
            return R.error("账号已被禁用,请联系管理员");
        }
        // 获取request
        HttpServletRequest request = HttpContextUtils.getHttpServletRequest();
        // 获取请求token
        String token = request.getHeader("Authorization");
        if (StringUtil.checkNotNull(token)) {
            token = token.replace("Bearer ", "");
            sysUserTokenService.saveToken(token, form.getUserId());
            // 用户权限列表
            Set<String> permissions = shiroService.getUserPermissions(form.getUserId());
            return R.ok().put("token", token).put("permissions", permissions);
        }
        // 记录登录日志
        saveLog("登录", "系统登录", "登录系统成功", "com.devplatform.admin.modules.sys.controller.SysLoginController.sso()", form, user, startTime);
        return R.error("Authorization Token 不能为空！");
    }

    /**
     * 退出
     */
    @SysLogMethod(operation = "注销", blockName = "系统退出", detail = "退出登录")
    @ApiOperation(value = "退出登录", notes = "退出登录")
    @PostMapping("/sys/logout")
    public R logout() {
        sysUserTokenService.logout(getUserId());
        return R.ok();
    }

    /**
     * 获取登录人信息,用于fegin调用
     */
    @ApiOperation(value = "获取登录人信息", notes = "获取登录人信息")
    @PostMapping("/sys/getCurrentUserInfo")
    public String getCurrentUserInfo() {
        Map<String, Object> currentLoginMap = new HashMap<String, Object>(10);
        currentLoginMap.put("userId", getUserId());
        currentLoginMap.put("getUser", getUser());
        currentLoginMap.put("currentUserName", getUser().getUsername());
        currentLoginMap.put("name", getUser().getName());
        currentLoginMap.put("roleIdList", getUser().getRoleIdList());
        currentLoginMap.put("mobile", getUser().getMobile());
        Gson g = new GsonBuilder().setDateFormat("yyyy-MM-dd HH:mm:ss").create();
        String jsonStr = g.toJson(R.ok(currentLoginMap));
        return jsonStr;
    }
}
