package com.devplatform.admin.modules.sys.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.devplatform.admin.common.utils.AbstractController;
import com.devplatform.admin.common.utils.Constants;
import com.devplatform.admin.modules.sys.bean.SysMenuEntity;
import com.devplatform.admin.modules.sys.bean.SysUserEntity;
import com.devplatform.admin.modules.sys.service.ShiroService;
import com.devplatform.admin.modules.sys.service.SysLogService;
import com.devplatform.admin.modules.sys.service.SysMenuService;
import com.devplatform.admin.modules.sys.service.SysRoleMenuService;
import com.devplatform.common.base.annotation.SysLog;
import com.devplatform.common.base.exception.RRException;
import com.devplatform.common.util.Constant;
import com.devplatform.common.util.R;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.*;

/**
 * 系统菜单
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2016年10月27日 下午9:58:15
 */
@Api(tags = {"系统菜单接口"})
@RestController
@RequestMapping("/sys/menu")
public class SysMenuController extends AbstractController {
  @Autowired private SysMenuService sysMenuService;
  @Autowired private ShiroService shiroService;
  @Autowired private SysRoleMenuService sysRoleMenuService;
  @Autowired private SysLogService sysLogService;

  /** 导航菜单 */
  @ApiOperation(value = "获取导航菜单", notes = "获取导航菜单")
  @GetMapping("/nav")
  public R nav(String stationId) {
    List<Map<String, Object>> menuList =
        sysMenuService.getUserMenuList(getUserId(), stationId);
    List<Map<String, Object>> nav = sysMenuService.getNav(getUserId(), stationId);
    Set<String> permissions = shiroService.getUserPermissions(getUserId());
    return R.ok().put("menuList", menuList).put("permissions", permissions).put("nav", nav);
  }

  /** 所有菜单列表 */
  @ApiOperation(value = "根据条件获取分页数据列表", notes = "根据条件获取分页数据列表")
  @ApiImplicitParam(
      name = "params",
      value = "查询参数",
      required = true,
      paramType = "query",
      dataType = "Map<String, Object>")
  @GetMapping("/list")
  public List<SysMenuEntity> list(@RequestParam Map<String, Object> params) {
    String stationId = (String) params.get("stationId");
    List<SysMenuEntity> menuList =
        sysMenuService.list(
            new LambdaQueryWrapper<SysMenuEntity>()
                .eq(StringUtils.isNotBlank(stationId), SysMenuEntity::getStationId, stationId));
    for (SysMenuEntity sysMenuEntity : menuList) {
      SysMenuEntity parentMenuEntity = sysMenuService.getById(sysMenuEntity.getParentId());
      if (parentMenuEntity != null) {
        sysMenuEntity.setParentName(parentMenuEntity.getName());
      }
    }

    return menuList;
  }

  /** 选择菜单(添加、修改菜单) */
  @ApiOperation(value = "选择菜单", notes = "选择菜单")
  @GetMapping("/select")
  public R select() {
    // 查询列表数据
    List<SysMenuEntity> menuList = sysMenuService.queryNotButtonList();

    // 添加顶级菜单
    SysMenuEntity root = new SysMenuEntity();
    root.setMenuId("0");
    root.setName("一级菜单");
    root.setParentId("-1");
    root.setOpen(true);
    menuList.add(root);

    return R.ok().put("menuList", menuList);
  }

  /** 菜单信息 */
  @ApiOperation(value = "根据ID获取菜单信息", notes = "根据ID获取菜单信息")
  @ApiImplicitParam(
      name = "menuId",
      value = "菜单主键",
      required = true,
      paramType = "query",
      dataType = "String")
  @GetMapping("/info/{menuId}")
  public R info(@PathVariable("menuId") String menuId) {
    SysMenuEntity menu = sysMenuService.getById(menuId);
    return R.ok().put("menu", menu);
  }

  /** 保存 */
  @ApiOperation(value = "保存菜单", notes = "保存菜单")
  @ApiImplicitParam(
      name = "menu",
      value = "菜单实体",
      required = true,
      paramType = "query",
      dataType = "SysMenuEntity")
  @PostMapping("/save")
  public R save(@RequestBody SysMenuEntity menu) {
    // 数据校验
    verifyForm(menu);
    menu.setSysSign(getUser().getSysSystem().getCode());
    sysMenuService.save(menu);
    return R.ok();
  }

  /** 修改 */
  @ApiOperation(value = "修改菜单", notes = "修改菜单")
  @ApiImplicitParam(
      name = "menu",
      value = "菜单实体",
      required = true,
      paramType = "query",
      dataType = "SysMenuEntity")
  @PostMapping("/update")
  public R update(@RequestBody SysMenuEntity menu) {
    // 数据校验
    verifyForm(menu);

    sysMenuService.updateById(menu);

    return R.ok();
  }

  /** 删除 */
  @ApiOperation(value = "删除菜单", notes = "删除菜单")
  @ApiImplicitParam(
      name = "menuId",
      value = "菜单实体",
      required = true,
      paramType = "query",
      dataType = "String")
  @PostMapping("/delete/{menuId}")
  public R delete(@PathVariable("menuId") String menuId) {
    if (Constants.STRING_1.equals(menuId)) {
      return R.error("系统菜单，不能删除");
    }

    // 判断是否有子菜单或按钮
    List<SysMenuEntity> menuList = sysMenuService.queryListParentId(menuId);
    if (menuList.size() > 0) {
      return R.error("请先删除子菜单或按钮");
    }

    sysMenuService.delete(menuId);

    return R.ok();
  }

  /** 删除 */
  @SysLog("查询菜单树")
  @ApiOperation(value = "查询菜单树", notes = "查询菜单树")
  @GetMapping("/getMenu")
  public R getMenu(String stationId) {
    List<Map<String, Object>> list =
        sysMenuService.listMaps(
            new LambdaQueryWrapper<SysMenuEntity>()
                .eq(SysMenuEntity::getStatus, 0)
//                .eq(StringUtils.isNotBlank(stationId), SysMenuEntity::getStationId, stationId)
                .orderByAsc(true, SysMenuEntity::getOrderNum));
    int size = list.size();
    List<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
    // 生成树结构
    Iterator<Map<String, Object>> iterator = list.iterator();
    while (iterator.hasNext()) {
      Map<String, Object> next = iterator.next();
      if ((Integer) next.get("type") == 0) {
        result.add(next);
        iterator.remove();
      }
    }
    // 给一级标签添加子集
    List<Map<String, Object>> maps = addChildren(list, result);
    return R.ok().put("result", maps).put("size", size - list.size());
  }

  /**
   * 添加子集
   * @param list
   * @param result
   * @return
   */
  public List<Map<String, Object>> addChildren(
      List<Map<String, Object>> list, List<Map<String, Object>> result) {
    for (Map<String, Object> firstMenu : result) {
      Iterator<Map<String, Object>> iterator = list.iterator();
      List<Map<String, Object>> array = new ArrayList<>();
      while (iterator.hasNext()) {
        Map<String, Object> menu = iterator.next();
        if (menu.get("parent_id").equals(firstMenu.get("menu_id"))) {
          array.add(menu);
          iterator.remove();
        }
      }
      firstMenu.put("children", array);
      if (array.size() > 0) {
        addChildren(list, array);
      }
    }
    return result;
  }

  /** 验证参数是否正确 */
  private void verifyForm(SysMenuEntity menu) {
    if (StringUtils.isBlank(menu.getName())) {
      throw new RRException("菜单名称不能为空");
    }

    if (menu.getParentId() == null) {
      throw new RRException("上级菜单不能为空");
    }

    // 菜单
    if (menu.getType() == Constant.MenuType.MENU.getValue()) {
      if (StringUtils.isBlank(menu.getUrl())) {
        throw new RRException("菜单URL不能为空");
      }
    }

    // 上级菜单类型
    int parentType = Constant.MenuType.CATALOG.getValue();
    if (!Constants.STRING_0.equals(menu.getParentId())) {
      SysMenuEntity parentMenu = sysMenuService.getById(menu.getParentId());
      parentType = parentMenu.getType();
    }

    // 目录、菜单
    if (menu.getType() == Constant.MenuType.CATALOG.getValue()
        || menu.getType() == Constant.MenuType.MENU.getValue()) {
      if (parentType != Constant.MenuType.CATALOG.getValue()) {
        throw new RRException("上级菜单只能为目录类型");
      }
      return;
    }

    // 按钮
    if (menu.getType() == Constant.MenuType.BUTTON.getValue()) {
      if (parentType != Constant.MenuType.MENU.getValue()) {
        throw new RRException("上级菜单只能为菜单类型");
      }
      return;
    }
  }

  /** 获取角色权限列表 */
  @SysLog("获取角色权限列表")
  @ApiOperation(value = "获取角色列表", notes = "获取角色列表")
  @GetMapping("/getRoleMenu")
  public R getRoleMenu(String roleId, String stationId) {
    List<HashMap<String, Object>> list = sysMenuService.getRoleMenu(roleId, stationId);
    return R.ok().put("list", list);
  }

  /** 修改角色权限列表 */
  @SysLog("修改角色权限列表")
  @ApiOperation(value = "修改角色权限列表", notes = "修改角色权限列表")
  @PostMapping("/updateRoleMenu")
  public R updateRoleMenu(@RequestBody Map<String, Object> para) {
    String roleId = (String) para.get("roleId");
    String stationId = (String) para.get("stationId");
    ArrayList<String> menuIds = (ArrayList<String>) para.get("menuIds");

    // 参数校验
    checkPara(roleId, menuIds);
    String code = getUser().getSysSystem().getCode();

    sysRoleMenuService.updateRoleMenu(code, roleId, menuIds, stationId);
    return R.ok();
  }

  /** 检查权限是否越权 */
  public void checkPara(String roleId, ArrayList<String> menuIds) {

    if (StringUtils.isBlank(roleId)) {
      throw new RRException("参数有误！");
    }
    // 只有超级管理员才能修改超级管理员角色
    //        if ("1".equals(roleId) && !"1".equals(getUserId())) {
    //            throw new RRException("没有权限操作！");
    //        }

    // 如果不是超级管理员，则需要判断角色的权限是否超过自己的权限
    //        if (Constant.SUPER_ADMIN.equals(getUserId())) {
    //            return;
    //        }
    // 查询用户所拥有的菜单列表
    //        List<String> menuIdList = sysUserService.queryAllMenuId(getUserId());
    // 判断是否越权
    //        if (!menuIdList.containsAll(menuIds)) {
    //            throw new RRException("新增角色的权限，已超出你的权限范围");
    //        }
  }

  /** 访问菜单时记录访问日志 */
  @ApiOperation(value = "访问菜单", notes = "访问菜单")
  @ApiImplicitParam(
      name = "params",
      value = "请求参数",
      required = true,
      paramType = "query",
      dataType = "Map<String, Object>")
  @PostMapping("/clickMenu")
  public R clickMenu(@RequestBody Map<String, Object> params) {
    Long startTime = System.currentTimeMillis();
    String operation = "查看";
    String menuName = params.get("menuName").toString();
    String detail = operation + menuName + "菜单";
    String stationId = params.get("stationId").toString();
    String pId = params.get("pId").toString();
    SysMenuEntity menuEntity = sysMenuService.getById(pId);
    SysUserEntity user = getUser();
    if (!stationId.equals(user.getSysSystem().getCode())) {
      sysLogService.saveLogRoutesSub(
          operation,
          menuEntity.getName(),
          detail,
          "com.devplatform.admin.modules.sys.controller.SysMenuController.clickMenu()",
          params,
          user,
          startTime,
          user.getSysSystem().getCode(),
          stationId);
    }
    return R.ok();
  }

  /** 访问菜单时记录访问日志 */
  @ApiOperation(value = "访问菜单", notes = "访问菜单")
  @ApiImplicitParam(
      name = "params",
      value = "请求参数",
      required = true,
      paramType = "query",
      dataType = "Map<String, Object>")
  @PostMapping("/clickMenuHeader")
  public R clickMenuHeader(@RequestBody Map<String, Object> params) {
    Long startTime = System.currentTimeMillis();
    String operation = "查看";
    String blockName = params.get("blockName").toString();
    String menuName = params.get("menuName").toString();
    String detail = operation + menuName + "菜单";
    String stationId = params.get("stationId").toString();
    SysUserEntity user = getUser();
    if (!stationId.equals(user.getSysSystem().getCode())) {
      sysLogService.saveLogRoutesSub(
          operation,
          blockName,
          detail,
          "com.devplatform.admin.modules.sys.controller.SysMenuController.clickMenuHeader()",
          params,
          user,
          startTime,
          user.getSysSystem().getCode(),
          stationId);
    }
    return R.ok();
  }
}
