package com.devplatform.admin.modules.sys.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.devplatform.admin.common.annotation.SysLogMethod;
import com.devplatform.admin.common.utils.AbstractController;
import com.devplatform.admin.common.utils.Constants;
import com.devplatform.admin.modules.sys.bean.SysRoleEntity;
import com.devplatform.admin.modules.sys.service.SysRoleMenuService;
import com.devplatform.admin.modules.sys.service.SysRoleService;
import com.devplatform.common.base.annotation.SysLog;
import com.devplatform.common.base.validator.ValidatorUtils;
import com.devplatform.common.util.PageUtils;
import com.devplatform.common.util.R;
import com.devplatform.common.util.StringUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 角色管理
 *
 * @author chenshun
 * @email sunlightcs@gmail.com
 * @date 2016年11月8日 下午2:18:33
 */
@Api(tags = {"角色管理接口"})
@RestController
@RequestMapping("/sys/role")
public class SysRoleController extends AbstractController {
  @Autowired private SysRoleService sysRoleService;
  @Autowired private SysRoleMenuService sysRoleMenuService;

  /** 角色列表 */
  @ApiOperation(value = "根据条件获取分页数据列表", notes = "根据条件获取分页数据列表")
  @ApiImplicitParam(
      name = "params",
      value = "查询参数",
      required = true,
      paramType = "query",
      dataType = "Map<String, Object>")
  @GetMapping("/list")
  public R list(@RequestParam Map<String, Object> params) {
    String roleName = (String) params.get("roleName");
    String createUserId = (String) params.get("createUserId");
    String stationId = (String) params.get("stationId");
    PageUtils page =
        sysRoleService.queryPage(
            params,
            new LambdaQueryWrapper<SysRoleEntity>()
                .like(StringUtil.checkNotNull(roleName), SysRoleEntity::getRoleName,roleName)
                .eq(StringUtils.isNotBlank(createUserId), SysRoleEntity::getCreateUserId, createUserId)
                .eq(StringUtils.isNotBlank(stationId), SysRoleEntity::getStationId, stationId)
                .orderByDesc(SysRoleEntity::getCreateTime));

    return R.ok().put("page", page);
  }

  /** 角色列表 */
  @ApiOperation(value = "获取角色列表", notes = "获取角色列表")
  @GetMapping("/select")
  public R select(String stationId) {
    Map<String, Object> map = new HashMap<>(5);
    map.put("deleted", 0);
    map.put("station_id", stationId);
    List<SysRoleEntity> list = (List<SysRoleEntity>) sysRoleService.listByMap(map);
    return R.ok().put("list", list);
  }

  /** 角色信息 */
  @ApiOperation(value = "根据ID获取角色信息", notes = "根据ID获取角色信息")
  @ApiImplicitParam(
      name = "roleId",
      value = "主键",
      required = true,
      paramType = "query",
      dataType = "String")
  @GetMapping("/info/{roleId}")
  public R info(@PathVariable("roleId") String roleId, String stationId) {
    SysRoleEntity role = sysRoleService.getById(roleId);

    // 查询角色对应的菜单
    List<String> menuIdList = sysRoleMenuService.queryMenuIdList(roleId, stationId);

    role.setMenuIdList(menuIdList);

    return R.ok().put("role", role);
  }

  /** 保存角色 */
  @SysLogMethod(operation = "新增", blockName = "系统管理", detail = "新增角色信息")
  @ApiOperation(value = "保存角色", notes = "保存角色")
  @ApiImplicitParam(
      name = "role",
      value = "角色实体类",
      required = true,
      paramType = "query",
      dataType = "SysRoleEntity")
  @PostMapping("/save")
  public R save(@RequestBody SysRoleEntity role) {
    ValidatorUtils.validateEntity(role);

    role.setCreateUserId(getUserId());
    role.setSysSign(getUser().getSysSystem().getCode());
    sysRoleService.saveMenu(role);

    return R.ok();
  }

  /** 修改角色 */
  @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "修改角色信息")
  @ApiOperation(value = "修改角色", notes = "修改角色")
  @ApiImplicitParam(
      name = "role",
      value = "角色实体类",
      required = true,
      paramType = "query",
      dataType = "SysRoleEntity")
  @PostMapping("/update")
  public R update(@RequestBody SysRoleEntity role) {
    ValidatorUtils.validateEntity(role);

    role.setCreateUserId(getUserId());
    role.setSysSign(getUser().getSysSystem().getCode());
    sysRoleService.update(role);

    return R.ok();
  }

  /** 删除角色 */
  @SysLogMethod(operation = "删除", blockName = "系统管理", detail = "删除角色信息")
  @ApiOperation(value = "删除角色", notes = "删除角色")
  @ApiImplicitParam(
      name = "roleIds",
      value = "角色主键(多个用','隔开)",
      required = true,
      paramType = "query",
      dataType = "String")
  @PostMapping("/delete")
  public R delete(@RequestBody String[] roleIds) {
    sysRoleService.deleteBatch(roleIds, getUser().getSysSystem().getCode());
    return R.ok();
  }

  /** 根据用户id查询角色信息 */
  @ApiOperation(value = "根据用户id查询角色信息", notes = "根据用户id查询角色信息")
  @ApiImplicitParam(
      name = "userId",
      value = "用户id",
      required = true,
      paramType = "query",
      dataType = "String")
  @GetMapping("/getUserRoleByUserId")
  public R getUserRoleByUserId(String userId) {
    if (StringUtils.isBlank(userId)) {
      return R.error("用户id不能为空");
    }
    List<SysRoleEntity> result = sysRoleService.queryUserRoleList(userId);
    return R.ok().put("data", result);
  }

  /** 配置角色信息 */
  @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "配置角色信息")
  @ApiOperation(value = "配置角色信息", notes = "配置角色信息")
  @ApiImplicitParams({
    @ApiImplicitParam(
        name = "userId",
        value = "用户id",
        required = true,
        paramType = "query",
        dataType = "String"),
    @ApiImplicitParam(
        name = "roleIds",
        value = "角色ids(多个角色id用逗号隔开)",
        required = true,
        paramType = "query",
        dataType = "String")
  })
  @PostMapping("/updateRoleList")
  public R updateRoleList(@RequestBody Map<String, Object> para) {
    String userId = (String) para.get("userId");
    String roleIds = (String) para.get("roleIds");
    String stationId = (String) para.get("stationId");
    if (StringUtils.isBlank(userId)) {
      return R.error("userId不能为空");
    }
    String code = getUser().getSysSystem().getCode();
    // 添加角色信息
    sysRoleService.addRole(code, userId, roleIds, stationId);
    return R.ok();
  }

  /** 获取角色信息 */
  @SysLog("获取角色列表")
  @ApiOperation(value = "获取角色列表", notes = "获取角色列表")
  @GetMapping("/getRoleList")
  public R getRoleList(String stationId) {
    QueryWrapper<SysRoleEntity> queryWrapper = new QueryWrapper<>();
    queryWrapper
        .eq("deleted", 0)
        .eq(StringUtils.isNotBlank(stationId), "station_id", stationId)
        .orderByDesc("create_time");
    List<SysRoleEntity> list = sysRoleService.list(queryWrapper);
    return R.ok().put("list", list);
  }

  /** 保存角色 */
  @SysLogMethod(operation = "新增", blockName = "系统管理", detail = "新增角色信息")
  @ApiOperation(value = "保存角色", notes = "保存角色")
  @ApiImplicitParam(
      name = "role",
      value = "角色实体类",
      required = true,
      paramType = "query",
      dataType = "SysRoleEntity")
  @PostMapping("/saveRole")
  public R saveRole(@RequestBody SysRoleEntity role) {
    ValidatorUtils.validateEntity(role);
    role.setCreateTime(new Date());
    role.setCreateUserId(getUserId());
    role.setSysSign(getUser().getSysSystem().getCode());
    sysRoleService.save(role);
    return R.ok();
  }

  /** 修改角色 */
  @SysLogMethod(operation = "修改", blockName = "系统管理", detail = "修改角色信息")
  @ApiOperation(value = "修改角色", notes = "修改角色")
  @ApiImplicitParam(
      name = "role",
      value = "角色实体类",
      required = true,
      paramType = "query",
      dataType = "SysRoleEntity")
  @PostMapping("/updateRole")
  public R updateRole(@RequestBody SysRoleEntity role) {
    ValidatorUtils.validateEntity(role);
    // 只有超级管理员才能修改
    if (Constants.STRING_1.equals(role.getRoleId()) && !Constants.STRING_1.equals(getUserId())) {
      return R.error("没有权限操作！");
    }
    role.setUpdateUserId(getUserId());
    role.setUpdateTime(new Date());
    role.setSysSign(getUser().getSysSystem().getCode());
    sysRoleService.updateRole(role);
    return R.ok();
  }

  /** 角色信息 */
  @ApiOperation(value = "根据ID获取角色信息", notes = "根据ID获取角色信息")
  @ApiImplicitParam(
      name = "roleId",
      value = "主键",
      required = true,
      paramType = "query",
      dataType = "String")
  @GetMapping("/getRoleInfo")
  public R getRoleInfo(String roleId) {
    if (StringUtils.isBlank(roleId)) {
      return R.error("参数有误！");
    }
    SysRoleEntity role = sysRoleService.getById(roleId);
    return R.ok().put("role", role);
  }
}
