package org.jeecg.modules.subwayNetwork.controller;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.jeecg.common.api.dto.PageSearch;
import org.jeecg.common.api.vo.Result;
import org.jeecg.common.aspect.annotation.AutoLog;
import org.jeecg.common.exception.JeecgBootException;
import org.jeecg.common.system.base.controller.JeecgController;
import org.jeecg.modules.subwayNetwork.dto.LightRailQueryDTO;
import org.jeecg.modules.subwayNetwork.entity.LightRail;
import org.jeecg.modules.subwayNetwork.entity.SectionStationMap;
import org.jeecg.modules.subwayNetwork.entity.TrainStation;
import org.jeecg.modules.subwayNetwork.service.ILightRailService;
import org.jeecg.modules.subwayNetwork.service.ISectionStationMapService;
import org.jeecg.modules.subwayNetwork.service.ITrainStationService;
import org.jeecg.modules.subwayNetwork.vo.LightRailQueryVO;
import org.jeecg.modules.subwayNetwork.vo.SectionStationMapVO;
import org.jeecg.modules.subwayNetwork.vo.SectionStationNode;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.*;

/**
 * @Description: 线路车站-轻轨线路
 * @Author: jeecg-boot
 * @Date: 2023-06-14
 * @Version: V1.0
 */
@Api(tags = "线路车站-轻轨线路")
@RestController
@RequestMapping("/subwayNetwork/lightRail")
@Slf4j
public class LightRailController extends JeecgController<LightRail, ILightRailService> {

    @Resource
    private ISectionStationMapService sectionStationMapService;

    @Resource
    private ITrainStationService trainStationService;

    @AutoLog(value = "线路车站-轻轨线路-分页列表查询")
    @ApiOperation(value = "线路车站-轻轨线路-分页列表查询", notes = "线路车站-轻轨线路-分页列表查询")
    @PostMapping(value = "/list")
    public Result<IPage<LightRailQueryVO>> queryPageList(@RequestBody PageSearch<LightRailQueryDTO> dto) {

        IPage<LightRailQueryVO> pageList = this.service.queryPageList(dto);
        return Result.OK(pageList);
    }


    @AutoLog(value = "线路车站-轻轨线路-编辑")
    @ApiOperation(value = "线路车站-轻轨线路-编辑", notes = "线路车站-轻轨线路-编辑")
    @PostMapping(value = "/edit")
    public Result<String> edit(@RequestBody LightRail lightRail) {
        //获取当前用户
        lightRail.setUpdateTime(new Date());
        if (ObjectUtil.isEmpty(lightRail.getId())) {
            lightRail.setCreateTime(new Date());
            this.service.save(lightRail);
        } else {
            this.service.updateById(lightRail);
        }

        return Result.OK("编辑成功!");
    }

    /**
     * 通过id删除
     *
     * @param id
     * @return
     */
    @AutoLog(value = "线路车站-轻轨线路-通过id删除")
    @ApiOperation(value = "线路车站-轻轨线路-通过id删除", notes = "线路车站-轻轨线路-通过id删除")
    @GetMapping(value = "/delete")
    public Result<String> delete(@RequestParam(name = "id", required = true) String id) {
        // 区间线路关联是否都删除了
        Long sectionStationMapCount = sectionStationMapService.lambdaQuery()
                .eq(SectionStationMap::getLightRailId, id)
                .count();
        if (sectionStationMapCount > 0) {
            throw new JeecgBootException("该线路下面还有关联区间");
        }

        // 车站关联是否删除
        Long trainStationCount = trainStationService.lambdaQuery()
                .eq(TrainStation::getLightRailId, id)
                .count();
        if (trainStationCount > 0) {
            throw new JeecgBootException("该线路下面还有车站");
        }
        this.service.removeById(id);

        return Result.OK("删除成功!");
    }

    /**
     * 批量删除
     *
     * @param ids
     * @return
     */
    @AutoLog(value = "线路车站-轻轨线路-批量删除")
    @ApiOperation(value = "线路车站-轻轨线路-批量删除", notes = "线路车站-轻轨线路-批量删除")
    @GetMapping(value = "/deleteBatch")
    public Result<String> deleteBatch(@RequestParam(name = "ids", required = true) String ids) {
        String message = "";
        Integer totalNum = 0;
        Integer successNum = 0;
        Integer failNum = 0;
        String failMessage = " ";
        // 1.查询待删除的线路
        List<LightRail> lightRails = this.service.lambdaQuery()
                .in(LightRail::getId, Arrays.asList(ids.split(",")))
                .list();


        // 2.循环删除和校验
        for (LightRail lightRail : lightRails) {
            totalNum++;


            Long sectionStationMapCount = sectionStationMapService.lambdaQuery()
                    .eq(SectionStationMap::getLightRailId, lightRail.getId())
                    .count();
            if (sectionStationMapCount > 0) {
                failNum++;
                failMessage = failMessage + lightRail.getRailLineName() + " ";
                continue;
            }
            Long trainStationCount = trainStationService.lambdaQuery()
                    .eq(TrainStation::getLightRailId, lightRail.getId())
                    .count();
            if (trainStationCount > 0) {
                failNum++;
                failMessage = failMessage + lightRail.getRailLineName() + " ";
                continue;
            }

            successNum++;
            this.service.removeById(lightRail.getId());
        }

        message = "选中" + totalNum + "条数据已删除数据" + successNum + "条";

        if (totalNum - successNum > 0) {
            message = message + ",未删除" + failNum + "条。未删除" + failMessage + "原因是线路下面有关联车站或区间故不能被删除";
        }
        return Result.batchDelete(message);
    }

    /**
     * 通过id查询
     *
     * @param id
     * @return
     */
    @AutoLog(value = "线路车站-轻轨线路-通过id查询")
    @ApiOperation(value = "线路车站-轻轨线路-通过id查询", notes = "线路车站-轻轨线路-通过id查询")
    @GetMapping(value = "/queryById")
    public Result<LightRail> queryById(@RequestParam(name = "id", required = true) String id) {
        LightRail lightRail = this.service.getById(id);
        if (lightRail == null) {
            return Result.error("未找到对应数据");
        }
        return Result.OK(lightRail);
    }

    @AutoLog(value = "线路车站-轻轨线路-获取全部线路")
    @ApiOperation(value = "线路车站-轻轨线路-获取全部线路", notes = "线路车站-轻轨线路-获取全部线路")
    @GetMapping(value = "/listAll")
    public Result<List<Map<String, Object>>> listAll() {
        LambdaQueryWrapper<LightRail> lambdaQuery = Wrappers.lambdaQuery();
        lambdaQuery.eq(LightRail::getStatus, 1);
        List<LightRail> pageList = this.service.list(lambdaQuery);

        List<Map<String, Object>> list = new ArrayList<>();
        for (LightRail lightRail : pageList) {
            Map<String, Object> map = new HashMap<>();
            map.put("label", lightRail.getRailLineName());
            map.put("value", lightRail.getId());
            list.add(map);
        }
        return Result.OK(list);
    }


    @AutoLog(value = "线路车站-轻轨线路-获取区间车站映射")
    @ApiOperation(value = "线路车站-轻轨线路-获取区间车站映射", notes = "线路车站-轻轨线路-获取区间车站映射")
    @GetMapping(value = "/getSectionStation")
    public Result<List<SectionStationMapVO>> getSectionStation() {
        return Result.OK(this.service.getSectionStation());
    }


    @AutoLog(value = "线路车站-轻轨线路-获取线路下面所有的(区间-站点)树")
    @ApiOperation(value = "线路车站-轻轨线路-获取线路下面所有的(区间-站点)树", notes = "线路车站-轻轨线路-获取线路下面所有的(区间-站点)树")
    @GetMapping(value = "/getTree")
    public Result<List<SectionStationNode>> getTree() {
        return Result.OK(this.service.getTree());
    }

}
