package org.jeecg.modules.subwayNetwork.controller;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.SecurityUtils;
import org.jeecg.common.api.vo.Result;
import org.jeecg.common.aspect.annotation.AutoLog;
import org.jeecg.common.exception.JeecgBootException;
import org.jeecg.common.system.base.controller.JeecgController;
import org.jeecg.common.system.vo.LoginUser;
import org.jeecg.modules.subwayNetwork.dto.LineAliasQueryDTO;
import org.jeecg.modules.subwayNetwork.entity.LineAlias;
import org.jeecg.modules.subwayNetwork.entity.SectionStationMap;
import org.jeecg.modules.subwayNetwork.entity.TrainStation;
import org.jeecg.modules.subwayNetwork.service.ILineAliasService;
import org.jeecg.modules.subwayNetwork.service.ISectionStationMapService;
import org.jeecg.modules.subwayNetwork.service.ITrainStationService;
import org.jeecg.modules.subwayNetwork.vo.LightAliasQueryVO;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.*;

/**
 * @Description: 线路车站-线别管理
 */
@Api(tags = "线路车站-线别管理")
@RestController
@RequestMapping("/subwayNetwork/lightAlias")
@Slf4j
public class LightAliasController extends JeecgController<LineAlias, ILineAliasService> {

    @Resource
    private ISectionStationMapService sectionStationMapService;

    @Resource
    private ITrainStationService trainStationService;

    @AutoLog(value = "线路车站-线别管理-分页列表查询")
    @ApiOperation(value = "线路车站-线别管理-分页列表查询", notes = "线路车站-线别管理-分页列表查询")
    @GetMapping(value = "/list")
    public Result<IPage<LightAliasQueryVO>> queryPageList(@RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
                                                          @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize,
                                                          LineAliasQueryDTO dto) {
        Page<LightAliasQueryVO> pageData = new Page<>(pageNo, pageSize);
        IPage<LightAliasQueryVO> pageList = this.service.queryPageList(pageData, dto);
        return Result.OK(pageList);
    }


    @AutoLog(value = "线路车站-线别管理-编辑")
    @ApiOperation(value = "线路车站-线别管理-编辑", notes = "线路车站-线别管理-编辑")
    @PostMapping(value = "/edit")
    public Result<String> edit(@RequestBody LineAlias lineAlias) {
        lineAlias.setUpdateTime(new Date());
        //获取当前登录人信息
        LoginUser user = (LoginUser) SecurityUtils.getSubject().getPrincipal();
        String realname = user.getRealname();
        lineAlias.setUpdateBy(realname);
        //获取当前用户
        if (ObjectUtil.isEmpty(lineAlias.getId())) {
            List<LineAlias> list = this.getQueryList(lineAlias.getLineAliasName());
            if (list != null && list.size() > 0) {
                return Result.error("线别名称已存在");
            }
            this.service.save(lineAlias);
        } else {
            LambdaQueryWrapper<LineAlias> lambdaQuery = Wrappers.lambdaQuery();
            lambdaQuery.ne(LineAlias::getId, lineAlias.getId());
            lambdaQuery.eq(LineAlias::getLineAliasName, lineAlias.getLineAliasName());
            List<LineAlias> list = this.service.list(lambdaQuery);
            if (list != null && list.size() > 0) {
                return Result.error("线别名称已重复");
            }
            this.service.updateById(lineAlias);
        }

        return Result.OK("编辑成功!");
    }

    /**
     * 查询线别
     *
     * @param name
     * @return
     */
    public List<LineAlias> getQueryList(String name) {
        LambdaQueryWrapper<LineAlias> lambdaQuery = Wrappers.lambdaQuery();
        lambdaQuery.eq(LineAlias::getLineAliasName, name);
        List<LineAlias> list = this.service.list(lambdaQuery);
        return list;
    }

    /**
     * 通过id删除
     *
     * @param id
     * @return
     */
    @AutoLog(value = "线路车站-线别管理-通过id删除")
    @ApiOperation(value = "线路车站-线别管理-通过id删除", notes = "线路车站-线别管理-通过id删除")
    @GetMapping(value = "/delete")
    public Result<String> delete(@RequestParam(name = "id", required = true) String id) {
        // 区间线路关联是否都删除了
        Long sectionStationMapCount = sectionStationMapService.lambdaQuery()
                .eq(SectionStationMap::getStationLineAliasId, id)
                .count();
        if (sectionStationMapCount > 0) {
            throw new JeecgBootException("该线别下面还有关联区间");
        }

        // 车站关联是否删除
        Long trainStationCount = trainStationService.lambdaQuery()
                .eq(TrainStation::getLineAliasId, id)
                .count();
        if (trainStationCount > 0) {
            throw new JeecgBootException("该线别下面还有车站");
        }
        this.service.removeById(id);


        this.service.removeById(id);
        return Result.OK("删除成功!");
    }

    /**
     * 批量删除
     *
     * @param ids
     * @return
     */
    @AutoLog(value = "线路车站-线别管理-批量删除")
    @ApiOperation(value = "线路车站-线别管理-批量删除", notes = "线路车站-线别管理-批量删除")
    @GetMapping(value = "/deleteBatch")
    public Result<String> deleteBatch(@RequestParam(name = "ids", required = true) String ids) {
        List<String> idList = Arrays.asList(ids.split(","));
        for (String id : idList) {
            Long sectionStationMapCount = sectionStationMapService.lambdaQuery()
                    .eq(SectionStationMap::getStationLineAliasId, id)
                    .count();
            if (sectionStationMapCount > 0) {
                throw new JeecgBootException("该线别下面还有关联区间");
            }

            Long trainStationCount = trainStationService.lambdaQuery()
                    .eq(TrainStation::getLineAliasId, id)
                    .count();
            if (trainStationCount > 0) {
                throw new JeecgBootException("该线别下面还有车站");
            }
        }

        this.service.removeByIds(idList);
        return Result.OK("批量删除成功!");
    }

    @AutoLog(value = "线路车站-线别管理-获取线别管理")
    @ApiOperation(value = "线路车站-线别管理-获取线别管理", notes = "线路车站-线别管理-获取线别管理")
    @GetMapping(value = "/listAll")
    public Result<List<Map<String, Object>>> listAll() {
        List<LineAlias> records = this.service.list();

        List<Map<String, Object>> list = new ArrayList<>();
        for (LineAlias alias : records) {
            Map<String, Object> map = new HashMap<>();
            map.put("label", alias.getLineAliasName());
            map.put("code", alias.getLineAliasCode());
            map.put("value", alias.getId());
            map.put("type", alias.getType());
            list.add(map);
        }
        return Result.OK(list);
    }

}
